/*
 * Copyright (c) 2017 - 2020 Jolla Ltd.
 * Copyright (c) 2020 Open Mobile Platform LLC.
 *
 * License: Proprietary
 */

#ifndef SAILFISH_MDM_CONNECTIVITY_H
#define SAILFISH_MDM_CONNECTIVITY_H

#include <QObject>
#include <QDateTime>
#include <QString>
#include <QStringList>
#include <QVector>

class NetworkService;

namespace Sailfish {
namespace Mdm {

class ConnectivitySettingsPrivate;
class Q_DECL_EXPORT ConnectivitySettings : public QObject
{
    Q_OBJECT
    Q_PROPERTY(bool ready READ ready NOTIFY readyChanged)
    Q_PROPERTY(bool flightModeEnabled READ flightModeEnabled WRITE setFlightModeEnabled NOTIFY flightModeEnabledChanged)
    Q_PROPERTY(bool wlanEnabled READ wlanEnabled WRITE setWlanEnabled NOTIFY wlanEnabledChanged)
    Q_PROPERTY(bool bluetoothEnabled READ bluetoothEnabled WRITE setBluetoothEnabled NOTIFY bluetoothEnabledChanged)

public:
    enum Mode {
        AsynchronousMode,
        SynchronousMode
    };

    explicit ConnectivitySettings(QObject *parent = Q_NULLPTR);
    explicit ConnectivitySettings(Mode mode, QObject *parent = Q_NULLPTR);

    bool ready() const;

    bool flightModeEnabled() const;
    void setFlightModeEnabled(bool value);

    bool wlanEnabled() const;
    void setWlanEnabled(bool value);

    bool bluetoothEnabled() const;
    void setBluetoothEnabled(bool value);

    bool isWifiScanAvailable() const;

    class MobileNetworkAccessPointSettings {
    public:
        enum Type {
            Default,
            Internet = Default,
            Mms,
            Ims
        };
        enum Protocol {
            IP,
            IPv4 = IP,
            IPv6,
            Dual
        };
        enum GprsAuthentication {
            None,
            PAP,
            CHAP,
            Any
        };

        Type type;
        QString connectionName;
        QString accessPointName;
        QString username;
        QString password;
        Protocol protocol;
        GprsAuthentication gprsAuthentication;
        QString messageCenter;
        QString messageProxy;

        bool isValid() const;
    };

    QStringList mobileNetworkImsis() const;
    QMap<MobileNetworkAccessPointSettings::Type, MobileNetworkAccessPointSettings> accessPointSettings(const QString &imsi) const;
    bool setAccessPointSettings(const QString &imsi, const MobileNetworkAccessPointSettings &settings);
    bool connectToMobileNetwork(const QString &imsi);
    bool disconnectFromMobileNetwork(const QString &imsi);
    bool isConnectedToMobileNetwork(const QString &imsi) const;

    enum CellularTechnology {
        TechnologyNone = 0,
        Technology2G   = 1,
        Technology3G   = 2,
        Technology4G   = 4,
        Technology5G   = 8,
    };
    Q_ENUM(CellularTechnology)
    Q_DECLARE_FLAGS(CellularTechnologies, CellularTechnology)
    Q_FLAG(CellularTechnologies)

    QStringList modemPaths() const;
    QString modemSimImsi(const QString &modemPath) const;
    QVector<CellularTechnologies> supportedCellularTechnologies(const QString &modemPath) const;
    CellularTechnologies selectedCellularTechnologies(const QString &modemPath) const;
    Q_INVOKABLE bool selectCellularTechnologies(const QString &modemPath, Sailfish::Mdm::ConnectivitySettings::CellularTechnologies techs);

    QVector<NetworkService*> getAvailableWifiServices() const;
    QVector<NetworkService*> getSavedWifiServices() const;

    void scanWifiServices();

signals:
    void readyChanged();
    void wlanEnabledChanged();
    void bluetoothEnabledChanged();
    void flightModeEnabledChanged();
    void wifiServicesChanged();
    void wifiScanAvailable();
    void wifiScanFinished();
    void wifiScanUnavailable();

private:
    ConnectivitySettingsPrivate *d_ptr;
    Q_DISABLE_COPY(ConnectivitySettings)
    Q_DECLARE_PRIVATE(ConnectivitySettings)
};

} // Mdm
} // Sailfish

Q_DECLARE_OPERATORS_FOR_FLAGS(Sailfish::Mdm::ConnectivitySettings::CellularTechnologies)

#endif // SAILFISH_MDM_CONNECTIVITY_H
