/* This file is part of the Aurora OS SDK
*/

var merSdkTargets = "AuroraOS-5.1.6.110-MB2-aarch64 AuroraOS-5.1.6.110-MB2-x86_64 AuroraOS-5.1.6.110-MB2-armv7hl";
var merSdkVmName = "Aurora Build Engine 5.1.6.110-mb2";
var merSdkVBoxUri = "sfdkvm:VirtualBox#" + merSdkVmName;
var merSdkVdiCapacityMB = "32000";
var merSdkVMSshPort = "2222";
var merSdkVMDBusPort = "7777";
var merSdkDir = "@TargetDir@/mersdk";
var sharedTargetsDir = "@TargetDir@/mersdk/targets";
var sharedSshDir = "@TargetDir@/mersdk/ssh";
var sharedConfigDir = "@TargetDir@/vmshare";
var sshKeyDir = sharedConfigDir + "/ssh/private_keys";

function dockerNetworkExists(networkName)
{
    const networkExistsCommand = ["network", "ls", "--filter", "name=" + networkName, "--quiet"];
    const result = installer.execute(installer.value("dockerPath"), networkExistsCommand)[0];
    return result.trim() !== "";
}

function dockerImageExists(imageName)
{
    const imageExistsCommand = ["image", "ls", imageName, "--quiet"];
    const result = installer.execute(installer.value("dockerPath"), imageExistsCommand)[0];
    return result.trim() !== "";
}

function componentExists(componentName)
{
    var retVal = false;
    installer.components().forEach(function (component) {
        if (component.name === componentName)
            retVal = true;
    });
    return retVal;
}

function dockerAvailable()
{
    return installer.value("dockerPath") && componentExists("org.merproject.mersdkdocker")
        && installer.value("os") !== "mac";
}

function dockerUsable()
{
    result = installer.execute(installer.value("dockerPath"), "ps");
    if (result.length < 2) {
        console.log("could not execute " + installer.value("dockerPath"));
        return false;
    }
    if (result[1] !== 0)
        console.log("docker returned " + result[1]);
    return result[1] === 0;
}

function vboxAvailable()
{
    return installer.value("vboxManagePath");
}

function setBuildEngineType(buildEngineType)
{
    installer.setValue("buildEngineType", buildEngineType);
    component.setValue("Dependencies", "");

    if (buildEngineType === "docker") {
        const merSdkDockerTag = installer.environmentVariable("USERNAME")
            || installer.environmentVariable("USER")
            || "latest";
        const merSdkDockerImage = "aurora-os-build-engine-5.1.6.110-mb2:" + merSdkDockerTag;
        installer.setValue("merSdkVmName", merSdkDockerImage);
        installer.setValue("merSdkVmUri", "sfdkvm:Docker#" + merSdkDockerImage);
        component.addDependency("org.merproject.mersdkdocker");
    } else {
        installer.setValue("merSdkVmName", merSdkVmName);
        installer.setValue("merSdkVmUri", merSdkVBoxUri);
        component.addDependency("org.merproject.mersdkvbox");
    }

    installer.componentsToInstallNeedsRecalculation();
}

function setupBuildEngineWidget()
{
    installer.addWizardPage(component, "BuildEngineTypeWidget",
            QInstaller.ComponentSelection);

    var buildEngineWidget = gui.findChild(component.userInterface("BuildEngineTypeWidget"),
            "scrollAreaWidgetContents");
    buildEngineWidget.dockerLabel_emulators.visible = false;
    buildEngineWidget.dockerWarningLabel.text =
                qsTranslate("installscript", "<html><head/><body><p><span style=' color:#ef2929;'>Install </span>"
                    + "<a href='https://docs.docker.com/desktop/'>"
                    + "<span style=' text-decoration: underline; color:#0000ff;'>Docker</span></a>"
                    + "<span style=' color:#ef2929;'> to enable this option</span></p></body></html>");
 
    if (installer.value("buildEngineType") === "docker")
        buildEngineWidget.dockerRadioButton.checked = true;
    else
        buildEngineWidget.vboxRadioButton.checked = true;

    if (!dockerAvailable()) {
        buildEngineWidget.dockerRadioButton.enabled = false;
        buildEngineWidget.dockerLabel.enabled = false;
        if (!componentExists("org.merproject.mersdkdocker")) {
            buildEngineWidget.dockerWarningLabel.text =
                qsTranslate("installscript", "<html><head/><body><p><span style='color:#ef2929;'>"
                    + "Feature available via Aurora SDK online repositories only.<br/>"
                    + "Please check your network connection and try again."
                    + "</span></p></body></html>");
            console.log(buildEngineWidget.dockerWarningLabel.text);
        }
    } else if (!dockerUsable()) {
        buildEngineWidget.dockerRadioButton.enabled = false;
        buildEngineWidget.dockerLabel.enabled = false;
        buildEngineWidget.dockerWarningLabel.text =
            qsTranslate("installscript", "<html><head/><body><p><span style='color:#ef2929;'>"
                + "Docker not running or misconfigured.<br/>"
                + "Please follow the instructions in the "
                + "<a href=\"https://developer.auroraos.ru/doc/sdk/app_development/setup\">Aurora SDK setup</a>"
                + " and try again"
                + "</span></p></body></html>");
    } else {
        buildEngineWidget.dockerWarningLabel.visible = false;
        if (installer.value("os") === "win")
            buildEngineWidget.dockerLabel_emulators.visible = true;
    }

    if (!vboxAvailable()) {
        buildEngineWidget.vboxRadioButton.enabled = false;
        buildEngineWidget.vboxLabel.enabled = false;
    } else {
        buildEngineWidget.vboxWarningLabel.visible = false;
    }

    buildEngineWidget.dockerRadioButton.toggled.connect(this, function(checked) {
        if (checked)
            setBuildEngineType("docker");
    });
    buildEngineWidget.vboxRadioButton.toggled.connect(this, function(checked) {
        if (checked)
            setBuildEngineType("vbox");
    });
}

function Component()
{
    console.log("Component constructor: " + component.name);
    if (installer.value("abortInstaller"))
        return;

    component.setValue("DisplayName", qsTranslate("installscript", "Build Engine"));
    component.setValue("Description", component.value("DisplayName"));

    installer.setValue("merSdkVdiCapacityMB", merSdkVdiCapacityMB);

    installer.setValue("merSdkTargets", merSdkTargets);
    installer.setValue("merSdkVMSshPort", merSdkVMSshPort);
    installer.setValue("merSdkVMDBusPort", merSdkVMDBusPort);

    const setDirs = function() {
        installer.setValue("merSdkSharedTargetsDir", sharedTargetsDir);
        installer.setValue("merSdkSharedSshDir", sharedSshDir);
        installer.setValue("merSdkSharedConfigDir", sharedConfigDir);
        installer.setValue("merSdkSshKeyDir", sshKeyDir);
        installer.setValue("merSdkSshKeyFile", sshKeyDir + "/sdk");
        installer.setValue("merSdkSshKeyPublicFile", sshKeyDir + "/sdk.pub");
    }
    setDirs();
    installer.valueChanged.connect(this, function(key, value) {
            if (key === "TargetDir")
                setDirs();
    });

    if (installer.value("vboxManagePath")) {
        const registeredVMs = installer.execute(installer.value("vboxManagePath"), ["list", "vms"])[0];
        if (registeredVMs.indexOf("\"" + merSdkVmName + "\"") !== -1 && !component.fromOnlineRepository) {
            installer.setValue("installer_errors", installer.value("installer_errors") + ";;;" +
               qsTranslate("installscript", "%1: There is already a <em>%2</em> VirtualBox machine. " +
                    "Please remove it before restarting this installation.")
                               .arg(component.value("DisplayName"))
                               .arg(merSdkVmName));
            installer.setValue("abortInstaller", true);
            installer.setValue("existingInstallationWarning", true)
        }
    }
    if (installer.fileExists(installer.value("homeDir") + "/VirtualBox VMs/" + merSdkVmName) && !component.fromOnlineRepository) {
        installer.setValue("installer_errors", installer.value("installer_errors") + ";;;" +
            qsTranslate("installscript", "%1: There is already a <em>%2</em> directory in the <em>VirtualBox VMs</em> folder. " +
                 "Please remove it before restarting this installation.")
                           .arg(component.value("DisplayName"))
                           .arg(merSdkVmName));
        installer.setValue("abortInstaller", true);
        installer.setValue("existingInstallationWarning", true)
    }

    if (installer.isInstaller()) {
        if (installer.value("rootLoaded")) {
            Component.prototype.ctor();
        } else {
            installer.valueChanged.connect(this, function(key, value) {
                if (key === "rootLoaded")
                    Component.prototype.ctor();
            });
        }
    } else {
        // updater and package manager case
        setBuildEngineType(installer.value("buildEngineType"));
    }
}

Component.prototype.ctor = function()
{
    if (installer.value("buildEngineType") === "vbox" && !vboxAvailable()) {
        installer.setValue("installer_errors", installer.value("installer_errors") + ";;;" +
                qsTranslate("installscript", "%1: You are trying to install VirtualBox build engine, but have no VirtualBox installed")
                .arg(component.value("DisplayName")));
        installer.setValue("abortInstaller", true);
        installer.setValue("existingInstallationWarning", true);
        return;
    }
    if (installer.value("buildEngineType") === "docker" && !dockerAvailable()) {
        installer.setValue("installer_errors", installer.value("installer_errors") + ";;;" +
                qsTranslate("installscript", "%1: You are trying to install Docker build engine, but have no Docker installed")
                .arg(component.value("DisplayName")));
        installer.setValue("abortInstaller", true);
        installer.setValue("existingInstallationWarning", true);
        return;
    }
    if (installer.value("buildEngineType") === "docker" && !dockerUsable()) {
        installer.setValue("installer_errors", installer.value("installer_errors") + ";;;" +
                qsTranslate("installscript", "%1: You are trying to install Docker build engine, but your Docker configuration is not compatible.")
                .arg(component.value("DisplayName")));
        installer.setValue("abortInstaller", true);
        installer.setValue("existingInstallationWarning", true);
        return;
    }

    if (!installer.value("buildEngineType")) {
        if (vboxAvailable()) {
            setBuildEngineType("vbox");
        } else {
            if (componentExists("org.merproject.mersdkdocker")) {
                setBuildEngineType("docker");
            } else {
                installer.setValue("installer_errors", installer.value("installer_errors") + ";;;" +
                    qsTranslate("installscript", "%1: The Docker build engine was selected to install, but online repository is not reachable.")
                    .arg(component.value("DisplayName")));
                installer.setValue("abortInstaller", true);
                return;
            }
        }
    } else {
        setBuildEngineType(installer.value("buildEngineType"));
    }

    // Currently we don't support docker on mac, so we don't show the widget.
    if (installer.value("os") !== "mac")
        setupBuildEngineWidget();
}

Component.prototype.createOperations = function()
{
    // Call the base createOperations
    component.createOperations();

    component.registerPathForUninstallation(sharedConfigDir, true);
    component.registerPathForUninstallation(sharedSshDir, true);

    component.addOperation("Mkdir", sharedConfigDir);
    component.addOperation("Mkdir", sharedSshDir);

    const sharedMerHomeDir = installer.value("sharedMerHomeDir");
    installer.performOperation("Mkdir", [sharedMerHomeDir]);

    if (installer.value("buildEngineType") === "docker")
        addDockerOperations();
    else
        addVBoxOperations();
}

function addVBoxOperations()
{
    const vboxManage = installer.value("vboxManagePath");
    const vdiFileName = "@TargetDir@/mersdk/mer.vdi";
    var portcountparam = "";
    var version = installer.execute(vboxManage, ["--version"])[0];
    version = version.substring(0,3); // 4.0, 4.1, 4.2, 4.3 etc.
    const floatver = parseFloat(version);

    if (floatver < 4.3)
        portcountparam = "--sataportcount";
    else
        portcountparam = "--portcount";

    const cpuCount = installer.value("cpuCount");

    var memorySizeMb = 512;
    if (cpuCount > 1)
        memorySizeMb = 4096;

    const sharedMerHomeDir = installer.value("sharedMerHomeDir");

    const errorMessage = "errormessage=" +  qsTranslate("installscript", "Please shutdown the \"%1\" VM").arg(merSdkVmName);

    const parameterLists = [
        [
            vboxManage, "setextradata",
                        "global",
                        "GUI/SuppressMessages",
                        "remindAboutAutoCapture,remindAboutWrongColorDepth,showRuntimeError.warning.HostAudioNotResponding,remindAboutMouseIntegrationOff"
        ],[
            vboxManage, "createvm",
                        "--name", merSdkVmName,
                        "--register",
                        "--basefolder", "@TargetDir@/mersdk",
            "UNDOEXECUTE",
            vboxManage, "unregistervm",
                        "--delete", merSdkVmName, errorMessage
        ], [
            vboxManage, "modifyvm", merSdkVmName,
                        "--description", qsTranslate("installscript", "ATTENTION: This virtual machine is managed by the Aurora SDK."
                            + " Do not try to alter it!")
        ], [
            vboxManage, "modifyvm", merSdkVmName,
                        "--ostype", "Linux26_64",
                        "--memory", memorySizeMb,
                        "--vram", "10",
                        "--ioapic", "on",
                        "--pae", "on",
                        "--nic1", "nat",
                        "--cableconnected1", "on",
                        "--nictype1", "virtio",
                        "--natpf1", "guestssh,tcp,127.0.0.1," + merSdkVMSshPort + ",,22",
                        "--natpf1", "guestdbus,tcp,127.0.0.1," + merSdkVMDBusPort + ",,777",
                        "--nic2", "intnet",
                        "--cableconnected2", "on",
                        "--intnet2", "@ConfigVariant@",
                        "--nictype2", "virtio",
                        "--macaddress2", "08005A11F155"
        ], [
            vboxManage, "storagectl", merSdkVmName,
                        "--name", "SATA",
                        "--add", "sata",
                        "--controller", "IntelAhci",
                        portcountparam, "1",
                        "--bootable", "on"
        ], [
            vboxManage, "internalcommands", "sethduuid",
                        vdiFileName
        ], [
            vboxManage, "storageattach", merSdkVmName,
                        "--storagectl", "SATA",
                        "--port", "1",
                        "--medium", vdiFileName,
                        "--type", "hdd",
            "UNDOEXECUTE",
            "{0,1}", vboxManage, "closemedium",
                        "disk", vdiFileName, errorMessage
        ], [
            vboxManage, "sharedfolder",
                        "add", merSdkVmName,
                        "--name", "install",
                        "--hostpath", "@TargetDir@",
                        "--readonly",
            "UNDOEXECUTE",
            "{0,1}", vboxManage, "sharedfolder",
                        "remove", merSdkVmName,
                        "--name", "install", errorMessage
        ], [
            vboxManage, "sharedfolder",
                        "add", merSdkVmName,
                        "--name", "home",
                        "--hostpath", sharedMerHomeDir,
                        "--readonly",
            "UNDOEXECUTE",
            "{0,1}", vboxManage, "sharedfolder",
                        "remove", merSdkVmName,
                        "--name", "home", errorMessage
        ], [
            vboxManage, "sharedfolder",
                        "add", merSdkVmName,
                        "--name", "targets",
                        "--hostpath", sharedTargetsDir,
            "UNDOEXECUTE",
            "{0,1}", vboxManage, "sharedfolder",
                        "remove", merSdkVmName,
                        "--name", "targets", errorMessage
        ], [
            vboxManage, "sharedfolder",
                        "add", merSdkVmName,
                        "--name", "ssh",
                        "--hostpath", sharedSshDir,
                        "--readonly",
            "UNDOEXECUTE",
            "{0,1}", vboxManage, "sharedfolder",
                        "remove", merSdkVmName,
                        "--name", "ssh", errorMessage
        ], [
                    vboxManage, "sharedfolder",
                                "add", merSdkVmName,
                                "--name", "config",
                                "--hostpath", sharedConfigDir,
                    "UNDOEXECUTE",
                    vboxManage, "sharedfolder",
                                "remove", merSdkVmName,
                                "--name", "ssh", errorMessage
        ], [
            vboxManage, "setextradata", merSdkVmName,
                        "VBoxInternal2/SharedFoldersEnableSymlinksCreate/src1", "1"
        ], [
            vboxManage, "setextradata", merSdkVmName,
                        "VBoxInternal2/SharedFoldersEnableSymlinksCreate/targets", "1"
        ], [
            vboxManage, "modifyvm", merSdkVmName,
                        "--natdnshostresolver1", "on"
        ], [
            vboxManage, "modifyvm", merSdkVmName,
                        "--cpus", cpuCount
        ], [
            vboxManage, "guestproperty", "set", merSdkVmName,
                        "/AuroraSDK/VM/Swap/SizeMb", "0"
        ]
    ];

    parameterLists.forEach(function(parameterList) {
        component.addOperation("Execute", parameterList);
    });

    component.addOperation("Execute", [vboxManage, "sharedfolder",
                                       "add", merSdkVmName,
                                       "--name", "src1",
                                       "--hostpath", installer.value("workspaceDir"),
                                       "UNDOEXECUTE",
                                       vboxManage, "sharedfolder",
                                       "remove", merSdkVmName,
                                       "--name", "src1", errorMessage]);
    component.addOperation("Execute", [vboxManage, "guestproperty",
                                       "set", merSdkVmName,
                                       "/SailfishSDK/ENV/SAILFISH_SDK_SRC1_MOUNT_POINT",
                                       installer.value("workspaceMountPoint")]);

    component.addOperation("VmShutdown",
                           merSdkVmName,
                           "when=undo");

}

function addDockerOperations()
{
    const noop = installer.value("os") === "win"
        ? [ "cmd", "/c", "exit" ]
        : [ "true" ];
    const docker = installer.value("dockerPath");

    const userName = installer.environmentVariable("USERNAME")
        || installer.environmentVariable("USER");
    const networkName = "aurora-sdk-5.1.6.110-mb2:" + userName;

    Array.prototype.flat1 = function() {
        return this.reduce(function(acc, val) { return acc.concat(val) }, []);
    };

    const networkCreateCommand = [docker, "network", "create", "--driver", "bridge", networkName];
    const networkRemoveCommand = [docker, "network", "rm", networkName];

    const networkExists = dockerNetworkExists(networkName);

    if (networkExists) {
        const deleteNetworkTitle = qsTranslate("installscript", "A conflict of Docker networks has occurred.");
        const deleteNetworkMessage = qsTranslate("installscript", "The Docker network with name %1 already exists. Delete this network and create it again?").arg(networkName);
        var deleteNetwork = QMessageBox.question(
            "Delete Network?",
            deleteNetworkTitle,
            deleteNetworkMessage,
            QMessageBox.Yes | QMessageBox.No
        );

        if (deleteNetwork === QMessageBox.Yes) {
            component.addOperation("Execute", networkRemoveCommand);
            component.addOperation("Execute", [networkCreateCommand, "UNDOEXECUTE", networkRemoveCommand].flat1());
        } else {
            component.addOperation("Execute", [noop, "UNDOEXECUTE", networkRemoveCommand].flat1());
        }
    } else {
        component.addOperation("Execute", [networkCreateCommand, "UNDOEXECUTE", networkRemoveCommand].flat1());
    }

    component.addOperation("AppendFile", merSdkDir + "/Dockerfile",
`FROM scratch
ADD aurora.tar /
ADD --chown="${installer.value("uid")}" srv.tar / 
LABEL SharedInstall="${installer.value("TargetDir").replace(/\\/g, '/')}" \
      SharedHome="${installer.value("sharedMerHomeDir").replace(/\\/g, '/')}" \
      SharedSrc="${installer.value("workspaceDir").replace(/\\/g, '/')}" \
      SharedTarget="${installer.value("merSdkSharedTargetsDir").replace(/\\/g, '/')}" \
      SharedConfig="${installer.value("merSdkSharedConfigDir").replace(/\\/g, '/')}" \
      SharedSsh="${installer.value("merSdkSharedSshDir").replace(/\\/g, '/')}" \
      SshPort="${merSdkVMSshPort}" \
      DBusPort="${merSdkVMDBusPort}"
RUN chown -R --from=mersdk "${installer.value("uid")}" /etc/mersdk /home/mersdk; \
    usermod -u "${installer.value("uid")}" mersdk
CMD ["/usr/bin/setarch", "x86_64", "/sbin/init"]
`);

    const merSdkDockerTag = installer.environmentVariable("USERNAME")
        || installer.environmentVariable("USER")
        || "latest";
    const merSdkDockerImageName = "aurora-os-build-engine-5.1.6.110-mb2:" + merSdkDockerTag;
    const merSdkDockerContainerName = "/aurora-os-build-engine-5.1.6.110-mb2_" + merSdkDockerTag;
    const merSdkDockerVolumeName = "aurora-sdk-5.1.6.110-mb2-" + merSdkDockerTag;

    const imageBuildCommand = [docker, "build", "--tag", "@merSdkVmName@", "--quiet", merSdkDir];
    const imageRemoveCommand = [docker, "image", "rm", "--force", merSdkDockerImageName];
    const containerRemoveCommand = [docker, "container", "rm", "--force", merSdkDockerContainerName];
    const volumeRemoveCommand = [docker, "volume", "rm", "--force", merSdkDockerVolumeName];

    const imageExists = dockerImageExists(merSdkDockerImageName);

    if (imageExists) {
        const deleteImageTitle = qsTranslate("installscript", "A conflict of Docker images has occurred");
        const deleteImageMessage = qsTranslate("installscript", "The Docker image with name %1 already exists. Delete this image and create it again?").arg(merSdkDockerImageName);
        var deleteImage = QMessageBox.question(
            "Delete Image?",
            deleteImageTitle,
            deleteImageMessage,
            QMessageBox.Yes | QMessageBox.No
        );

        if (deleteImage === QMessageBox.Yes) {
            component.addOperation("Execute", containerRemoveCommand);
            component.addOperation("Execute", volumeRemoveCommand);
            component.addOperation("Execute", imageRemoveCommand);
            component.addOperation("Execute", [imageBuildCommand, "UNDOEXECUTE", imageRemoveCommand].flat1());
        } else {
            component.addOperation("Execute", [noop, "UNDOEXECUTE", imageRemoveCommand].flat1());
        }
    } else {
        component.addOperation("Execute", [imageBuildCommand, "UNDOEXECUTE", imageRemoveCommand].flat1());
    }

    component.addOperation("Execute", [noop,
                                       "UNDOEXECUTE",
                                       volumeRemoveCommand].flat1());

    component.addOperation("Execute", [noop,
                                       "UNDOEXECUTE",
                                       containerRemoveCommand].flat1());

    component.addOperation("Delete", merSdkDir + "/aurora.tar");
    component.addOperation("Delete", merSdkDir + "/Dockerfile");

    // Run docker with volume optin for initialize volume. The operation can take a long time, so we perform it during installation
    const engineImageRunCommand = [docker, "run", "--rm", "-v", merSdkDockerVolumeName + ":/srv/mer", merSdkDockerImageName, "echo"];
    component.addOperation("Execute", engineImageRunCommand);
}

Component.prototype.createOperationsForArchive = function(archive)
{
    if (installer.value("buildEngineType") === "docker") {
        if (archive.endsWith("docker.7z"))
            component.addOperation("Extract", archive, "@TargetDir@");
    } else {
        if (archive.endsWith("vbox.7z"))
            component.addOperation("Extract", archive, "@TargetDir@");
    }
}
