/****************************************************************************************
**
** SPDX-FileCopyrightText: 2013-2019 Jolla Ltd.
** SPDX-FileCopyrightText: 2021-2025 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
**
****************************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0
import ".."

SilicaItem {
    id: indicators

    property PageStack pageStack
    property real itemWidth: Theme.pageStackIndicatorWidth
    property real maxOpacity: pageStack.opacity
    property bool animatingPosition
    property real leftWidth: itemWidth
    readonly property bool backIndicatorDown: backMouseArea.pressed && backMouseArea.containsMouse
    property bool backIndicatorHighlighted: backIndicatorDown
    readonly property bool forwardIndicatorDown: forwardMouseArea.pressed && forwardMouseArea.containsMouse
    property bool forwardIndicatorHighlighted: forwardIndicatorDown
    property bool clickablePageIndicators: pageStack.currentPage ? pageStack.currentPage._clickablePageIndicators : true
    property bool compact: true

    onForwardIndicatorHighlightedChanged: {
        if (!forwardIndicatorHighlighted) {
            forwardPageIndicator.setProgress(forwardPageIndicator.progress)
            return
        }

        if (!(forwardPageIndicator.canNavigateForward && clickablePageIndicators && maxOpacity > 0.0)) {
            return
        }

        forwardPageIndicator.setProgress(1)
    }
    onBackIndicatorHighlightedChanged: {
        if (!backIndicatorHighlighted) {
            backPageIndicator.setProgress(backPageIndicator.progress)
            return
        }

        if (!(clickablePageIndicators && maxOpacity > 0.0 && backPageIndicator.backNavigation)) {
            return
        }

        backPageIndicator.setProgress(1)
    }

    opacity: enabled ? Math.min(maxOpacity, pageStack.opacity) : 0.0
    Behavior on opacity { FadeAnimation {} }

    height: pageStack._orientationTransposed ? parent.width : parent.height
    width: pageStack._orientationTransposed ? parent.height : parent.width

    property int currentNavigation: PageNavigation.NoNavigation
    property real currentLateralOffset: pageStack._currentContainer && pageStack._currentContainer.horizontalNavigationStyle ? pageStack._currentContainer.dragOffset : 0
    property bool busy: pageStack.busy
    onBusyChanged: {
        if (!busy) {
            currentNavigation = PageNavigation.NoNavigation
            partnerPageIndicator.x = -itemWidth/2
        }
    }

    onCurrentLateralOffsetChanged: {
        if (currentLateralOffset < 0) {
            currentNavigation = PageNavigation.Forward
        } else if (currentLateralOffset > 0) {
            currentNavigation = PageNavigation.Back
        } else {
            currentNavigation = PageNavigation.NoNavigation
        }
    }

    Connections {
        target: pageStack

        onBusyChanged: {
            backPageIndicator.setProgress(0)
            forwardPageIndicator.setProgress(0)
        }
    }

    Item {
        id: partnerPageIndicator

        property Item container: pageStack._currentContainer ? pageStack._currentContainer.transitionPartner : null
        property bool forwardNavigation: container && (container.page && container.page.forwardNavigation || container.attachedContainer)
        property bool canNavigateForward: container && (container.page && container.page.canNavigateForward || container.attachedContainer)
        property bool backNavigation: container && container.page && container.page.backNavigation && container.page._horizontalNavigationStyle && container.pageStackIndex !== 0
        property real lateralOffset: container && container.horizontalNavigationStyle ? container.dragOffset : 0

        onLateralOffsetChanged: {
            if (currentNavigation == PageNavigation.Forward) {
                x = lateralOffset-itemWidth/2 + indicators.width
            } else if (currentNavigation == PageNavigation.Back) {
                x = lateralOffset-itemWidth/2
            } else {
                x = -itemWidth/2
            }
        }

        height: indicators.height
        width: itemWidth

        visible: container && container.visible
        opacity: backNavigation && horizontalNavigationStyle && currentNavigation == PageNavigation.Back
                 ? 1.0
                 : (forwardNavigation && currentNavigation == PageNavigation.Forward ? (canNavigateForward ? 1.0 : Theme.opacityHigh) : 0.0)

        PageStackChevronIndicator {
            anchors.centerIn: parent
            compact: indicators.compact
        }
    }

    Item {
        id: backPageIndicator

        property Item container: pageStack._currentContainer
        property bool backNavigation: container && container.page
                                      && container.page.backNavigation && container.page._horizontalNavigationStyle && container.pageStackIndex !== 0
        property real progress: Math.min(1.0, pageStack._leftFlickDifference / pageStack._flickThreshold)

        function setProgress(progress) {
            backIndicatorAnimation.to = progress
            backIndicatorAnimation.restart()
        }

        onProgressChanged: {
            if (backMouseArea.containsMouse) {
                return
            }

            if (backIndicatorAnimation.running) {
                if (backChevronIndicator.progress > progress) {
                    return
                }
                backIndicatorAnimation.stop()
            }

            if (progress == 0 && backChevronIndicator.progress == 1) {
                backIndicatorAnimation.to = 0
                backIndicatorAnimation.restart()
                return
            }

            backChevronIndicator.progress = progress
        }

        x: currentLateralOffset
        height: indicators.height
        width: itemWidth

        // fade out as we approach the right edge
        opacity: !backNavigation
                 ? 0
                 : partnerPageIndicator.forwardNavigation
                   ? (partnerPageIndicator.canNavigateForward ? 1.0 : Theme.opacityHigh)
                   : (container
                      ? Math.min((indicators.width - currentLateralOffset) / (indicators.width * 0.25), 1.0)
                      : 0.0)

        Behavior on opacity {
            FadeAnimation {}
        }

        MouseArea {
            id: backMouseArea

            onContainsMouseChanged: parent.setProgress(containsMouse ? 1 : parent.progress)
            onClicked: pageStack.navigateBack()

            anchors.fill: parent
            enabled: indicators.clickablePageIndicators && indicators.maxOpacity > 0.0 && backPageIndicator.backNavigation
            cursorShape: enabled ? Qt.PointingHandCursor : Qt.ArrowCursor
        }

        NumberAnimation {
            id: backIndicatorAnimation

            target: backChevronIndicator
            property: "progress"
            duration: 200
            easing.type: Easing.InOutQuart
        }

        PageStackChevronIndicator {
            id: backChevronIndicator

            onProgressChanged: {
                if (backIndicatorAnimation.running) {
                    if (progress > parent.progress) {
                        return
                    }
                    backIndicatorAnimation.stop()
                }
            }

            anchors.centerIn: parent
            anchors.horizontalCenterOffset: Theme.dp(3)
            compact: indicators.compact
            opacity: indicators.clickablePageIndicators ? 1.0 : Theme.opacityHigh
        }
    }

    Item {
        id: forwardPageIndicator

        property Item container: pageStack._currentContainer
        property bool forwardNavigation: container && container.page && container.page.forwardNavigation
        property bool canNavigateForward: container && container.page && container.page.canNavigateForward
        property real progress: Math.min(1.0, pageStack._rightFlickDifference / pageStack._flickThreshold)

        function updateOpacity() {
            // if back navigation is not availble in transitionPartner, then fade out as we near the left edge
            var backNavOpacity = partnerPageIndicator.backNavigation
                    ? 1.0 : Math.min(1.0, (currentLateralOffset+indicators.width)/(indicators.width/4))
            backNavOpacity *= canNavigateForward ? 1.0 : Theme.opacityHigh
            if (currentLateralOffset == 0.0 && !pageStack.busy) {
                fadeAnim.to = container && forwardNavigation ? backNavOpacity : 0.0
                fadeAnim.restart()
            } else {
                fadeAnim.stop()
                opacity = container && forwardNavigation ? backNavOpacity : 0.0
            }
        }

        function setProgress(progress) {
            forwardIndicatorAnimation.to = progress
            forwardIndicatorAnimation.restart()
        }

        onProgressChanged: {
            if (forwardMouseArea.containsMouse) {
                return
            }

            if (forwardIndicatorAnimation.running) {
                if (forwardChevronIndicator.progress > progress) {
                    return
                }

                forwardIndicatorAnimation.stop()
            }


            if (progress == 0 && forwardChevronIndicator.progress == 1) {
                forwardIndicatorAnimation.to = 0
                forwardIndicatorAnimation.restart()
                return
            }

            forwardChevronIndicator.progress = progress
        }

        x: currentLateralOffset + indicators.width - itemWidth
        height: indicators.height
        width: itemWidth
        opacity: 0.0
        visible: container && container.visible

        onContainerChanged: updateOpacity()
        onForwardNavigationChanged: updateOpacity()
        onCanNavigateForwardChanged: updateOpacity()
        onXChanged: updateOpacity()

        MouseArea {
            id: forwardMouseArea

            onContainsMouseChanged: parent.setProgress(containsMouse ? 1 : parent.progress)
            onClicked: pageStack.navigateForward()

            anchors.fill: parent
            enabled: forwardPageIndicator.canNavigateForward && indicators.clickablePageIndicators && indicators.maxOpacity > 0.0
            cursorShape: enabled ? Qt.PointingHandCursor : Qt.ArrowCursor
        }

        FadeAnimation {
            id: fadeAnim
            target: forwardPageIndicator
        }

        NumberAnimation {
            id: forwardIndicatorAnimation

            target: forwardChevronIndicator
            property: "progress"
            duration: 200
            easing.type: Easing.InOutQuart
        }

        PageStackChevronIndicator {
            id: forwardChevronIndicator

            onProgressChanged: {
                if (forwardIndicatorAnimation.running) {
                    if (progress > parent.progress) {
                        return
                    }

                    forwardIndicatorAnimation.stop()
                }
            }

            anchors.centerIn: parent
            anchors.horizontalCenterOffset: -Theme.dp(3)
            progress: parent.progress
            rotation: 180
            compact: indicators.compact
            opacity: indicators.clickablePageIndicators ? 1.0 : Theme.opacityHigh
        }
    }

    transform: [
        Rotation {
            angle: pageStack._currentOrientationAngle
        }, Translate {
            x: {
                if (pageStack._currentOrientationAngle === 90) {
                    return height
                } else if (pageStack._currentOrientationAngle === 180) {
                    return width
                }
                return 0
            }
            y: {
                if (pageStack._currentOrientationAngle === 270) {
                    return width
                } else if (pageStack._currentOrientationAngle === 180) {
                    return height
                }
                return 0
            }
        }
    ]
    // For testing
    function _forwardPageIndicator() {
        return forwardPageIndicator
    }

    function _backPageIndicator() {
        return backPageIndicator
    }
}
