/**
 * SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#ifndef MARKET_LIB_RESULT_HPP
#define MARKET_LIB_RESULT_HPP

#include "globals.hpp"

#include <memory>
#include <string>

namespace Market {

class ErrorResultPrivate;
class MARKET_LIB_EXPORT_SYMBOL ErrorResult
{
public:
    ErrorResult(std::string error); /* NOLINT(google-explicit-constructor) */
    const std::string &Get() const;

private:
    std::shared_ptr<ErrorResultPrivate> m_pimpl;
};

template<typename T>
class SuccessResultPrivate;

template<typename T>
class MARKET_LIB_EXPORT_SYMBOL SuccessResult
{
public:
    SuccessResult(T success); /* NOLINT(google-explicit-constructor) */
    const T &Get() const;

private:
    std::shared_ptr<SuccessResultPrivate<T>> m_pimpl;
};

template<>
class MARKET_LIB_EXPORT_SYMBOL SuccessResult<void>
{};

template<typename T>
class ResultPrivate;

template<>
class ResultPrivate<void>;

/**
 * @brief Class that contains a value of type T or an error.
 * @class Result result.hpp <market/result.hpp>
 * @tparam T Type of possible contained value
 */
template<typename T>
class MARKET_LIB_EXPORT_SYMBOL Result
{
public:
    Result(ErrorResult error);        /* NOLINT(google-explicit-constructor) */
    Result(SuccessResult<T> success); /* NOLINT(google-explicit-constructor) */

    /**
     * @brief Checks whether the result contains an error.
     *
     * @return True if contains an error otherwise false.
     */
    bool HasError() const;

    /**
     * @brief Checks whether the result contains an error.
     *
     * @code{.cpp}
     * Result<Operation> operation = Operation::FromFile(...);
     *
     * if (!operation) {
     *     std::cout << "Couldn't read operation from file" << std::endl;
     * }
     * @endcode
     */
    explicit operator bool() const;

    /**
     * @brief Get the value that is stored in the result.
     *
     * @note This method raise an exception if the result contains an error.
     *
     * @return The value that is stored in the result.
     */
    const T &Get() const;

    /**
     * @brief Get pointer to the value that is stored in the result.
     *
     * @note This method raise an exception if the result contains an error.
     *
     * @return Pointer to the value that is stored in the result.
     */
    const T *operator->() const;

    /**
     * @brief Get the error that is stored in the result.
     *
     * @note This method raise an exception if the result doesn't contain an error.
     *
     * @return The error that is stored in the result.
     */
    const std::string &GetError() const;

private:
    std::shared_ptr<ResultPrivate<T>> m_pimpl;
};

/**
 * @brief A class that can contain an error and indicates that an error has occurred.
 * @class Result<void> result.hpp <market/result.hpp>
 */
template<>
class MARKET_LIB_EXPORT_SYMBOL Result<void>
{
public:
    Result(ErrorResult error);           /* NOLINT(google-explicit-constructor) */
    Result(SuccessResult<void> success); /* NOLINT(google-explicit-constructor) */

    /**
     * @brief Checks whether the result contains an error.
     *
     * @return True if contains an error otherwise false.
     */
    bool HasError() const;

    /**
     * @brief Checks whether the result contains an error.
     *
     * @code{.cpp}
     * Result<Operation> operation = Operation::FromFile(...);
     *
     * if (!operation) {
     *     std::cout << "Couldn't read operation from file" << std::endl;
     * }
     * @endcode
     */
    explicit operator bool() const;

    /**
     * @brief Get the error that is stored in the result.
     *
     * @note This method raise an exception if the result doesn't contain an error.
     *
     * @return The error that is stored in the result.
     */
    const std::string &GetError() const;

private:
    std::shared_ptr<ResultPrivate<void>> m_pimpl;
};

} /* namespace Market */

#endif /* MARKET_LIB_RESULT_HPP */
