/**
 * SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#ifndef MARKET_LIB_OPERATION_HPP
#define MARKET_LIB_OPERATION_HPP

#include "globals.hpp"
#include "result.hpp"

#include "common/error-code-enum.hpp"
#include "common/operation-status.hpp"
#include "common/operation-type.hpp"

#include <memory>
#include <string>

namespace Market {

/**
 * @brief Possible operation error codes
 */
using OperationErrorCode = Common::ErrorCode;

/**
 * @brief Possible operation statuses
 */
using OperationStatus = Common::OperationStatus;

/**
 * @brief Possible operation types
 */
using OperationType = Common::OperationType;

/**
 * @brief Unique operation identifier in string format
 */
using OperationUUID = std::string;

class OperationPrivate;

/**
 * @brief Class for getting information about the operations.
 * @class Operation operation.hpp <market/operation.hpp>
 */
class MARKET_LIB_EXPORT_SYMBOL Operation
{
public:
    Operation();
    Operation(OperationUUID uuid,
              std::string packageID,
              OperationType type,
              OperationStatus status,
              OperationErrorCode errorCode);

    /**
     * @brief Get the unique operation identifier.
     *
     * @return Unique operation identifier.
     */
    const OperationUUID &GetUUID() const;

    /**
     * @brief Get the unique operation identifier.
     *
     * @return Unique operation identifier.
     */
    const std::string &GetPackageID() const;

    /**
     * @brief Get the operation type.
     *
     * @return Operation type.
     */
    const OperationType &GetType() const;

    /**
     * @brief Get the operation status.
     *
     * @return Operation status.
     */
    const OperationStatus &GetStatus() const;

    /**
     * @brief Get the operation error code.
     *
     * @return Operation error code.
     */
    const OperationErrorCode &GetErrorCode() const;

    /**
     * @brief Read an operation serialized using the @ref SaveToFile()
     *        function from a file.
     *
     * @note This method may return an error result if deserialization
     *       failed or the file does not exist.
     *
     * @param path Path to file with serialized operation.
     *
     * @return @ref Result containing the deserialized operation or error.
     */
    static Result<Operation> FromFile(const std::string &path);

    /**
     * @brief Serialize the operation to a file to read it in the future
     *        using the @ref FromFile() method.
     *
     * @note This method may return an error result if serialization
     *       failed or any file system error has occurred.
     *
     * @param path Path where you want to save the operation.
     *
     * @return @ref Result of serialization or error.
     */
    Result<void> SaveToFile(const std::string &path) const;

private:
    std::shared_ptr<OperationPrivate> m_pimpl;
};

} /* namespace Market */

#endif /* MARKET_LIB_OPERATION_HPP */
