/**
 * SPDX-FileCopyrightText: Copyright 2024-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#ifndef COMMON_ERROR_CODE_ENUM_HPP
#define COMMON_ERROR_CODE_ENUM_HPP

#ifdef __cplusplus
#define ERROR_CODE_ENUM_ENTRY(CPP, C) CPP
#else
#define ERROR_CODE_ENUM_ENTRY(CPP, C) ERROR_CODE_##C
#endif

#ifdef __cplusplus
/**
 * @brief General data types and enums
 * @namespace Common
 */
namespace Common {
#endif

/**
 * @brief Error codes that the market daemon can return during the operation.
 * @enum ErrorCode
 */
enum ErrorCode {
    /** No error */
    ERROR_CODE_ENUM_ENTRY(None, NONE) = 0,
    /** Filesystem error occurred */
    ERROR_CODE_ENUM_ENTRY(Filesystem, FILESYSTEM),
    /** Internal error occurred */
    ERROR_CODE_ENUM_ENTRY(Internal, INTERNAL),
    /** The installation package path does not exist */
    ERROR_CODE_ENUM_ENTRY(FileNotFound, FILE_NOT_FOUND),
    /** Invalid package (package validation failed) */
    ERROR_CODE_ENUM_ENTRY(BadPackage, BAD_PACKAGE) __attribute__((deprecated)),
    /** The package has an invalid ID */
    ERROR_CODE_ENUM_ENTRY(BadPackageID, BAD_PACKAGE_ID),
    /** The package architecture does not match the system architecture */
    ERROR_CODE_ENUM_ENTRY(BadPackageArchitecture, BAD_PACKAGE_ARCHITECTURE),
    /** Invalid package signature */
    ERROR_CODE_ENUM_ENTRY(BadPackageSignature, BAD_PACKAGE_SIGNATURE) __attribute__((deprecated)),
    /** Attempt to downgrade the package */
    ERROR_CODE_ENUM_ENTRY(DowngradeAttempt, DOWNGRADE_ATTEMPT),
    /** The package upgrade has already been prepared (application is expected to close) */
    ERROR_CODE_ENUM_ENTRY(UpgradeAlreadyPrepared, UPGRADE_ALREADY_PREPARED)
        __attribute__((deprecated)),
    /** The package does not exist */
    ERROR_CODE_ENUM_ENTRY(PackageNotExist, PACKAGE_NOT_EXIST),
    /** Attempt to retrieve data about a corrupted package */
    ERROR_CODE_ENUM_ENTRY(CorruptedInstalledPackage, CORRUPTED_INSTALLED_PACKAGE),
    /** Failed to install package dependencies */
    ERROR_CODE_ENUM_ENTRY(UnresolvedDependency, UNRESOLVED_DEPENDENCY) __attribute__((deprecated)),
    /** The process is not authorized to perform the action */
    ERROR_CODE_ENUM_ENTRY(UnauthorizedAccess, UNAUTHORIZED_ACCESS),
    /** Attempt to install a package with system developer signature */
    ERROR_CODE_ENUM_ENTRY(SystemPackageInstallationAttempt, SYSTEM_PACKAGE_INSTALLATION_ATTEMPT),
    /** Installation of packages without a source signature is not allowed */
    ERROR_CODE_ENUM_ENTRY(PackagesWithoutSourceNotAllowed, PACKAGES_WITHOUT_SOURCE_NOT_ALLOWED),
    /** The administrator rejected the request to add a source to the trusted list */
    ERROR_CODE_ENUM_ENTRY(SourceRejectedByAdministrator, SOURCE_REJECTED_BY_ADMINISTRATOR),
    /** The current user has no rights to add a source to the trusted list */
    ERROR_CODE_ENUM_ENTRY(UserNotAuthorizedToManageTrustedSources,
                          USER_NOT_AUTHORIZED_TO_TRUST_THE_SOURCE),
    /** The package does not have a developer's signature */
    ERROR_CODE_ENUM_ENTRY(MissingDeveloperSignature, MISSING_DEVELOPER_SIGNATURE),
    /** The developer of the already installed package does not match the developer of the package being installed */
    ERROR_CODE_ENUM_ENTRY(DeveloperMismatch, DEVELOPER_MISMATCH),
    /** The package is signed with a developer key that has been superseded */
    ERROR_CODE_ENUM_ENTRY(DeveloperKeySuperseded, DEVELOPER_KEY_SUPERSEDED),
    /** The package is signed with a developer key that has been blacklisted */
    ERROR_CODE_ENUM_ENTRY(DeveloperKeyBlacklisted, DEVELOPER_KEY_BLACKLISTED),
    /** The package is signed with an illegal developer certificate */
    ERROR_CODE_ENUM_ENTRY(IllegalDeveloperCertificate, ILLEGAL_DEVELOPER_CERTIFICATE),
    /** The package is signed with an illegal source certificate */
    ERROR_CODE_ENUM_ENTRY(IllegalSourceCertificate, ILLEGAL_SOURCE_CERTIFICATE),
    /** The package version has an incorrect value */
    ERROR_CODE_ENUM_ENTRY(BadPackageVersion, BAD_PACKAGE_VERSION),
    /** The package did not pass content validation */
    ERROR_CODE_ENUM_ENTRY(BadPackageContent, BAD_PACKAGE_CONTENT),
    /** Failed to parse the package or read the required package data */
    ERROR_CODE_ENUM_ENTRY(BadPackageFileFormat, BAD_PACKAGE_FILE_FORMAT),
    /** The package has a disallowed organization name */
    ERROR_CODE_ENUM_ENTRY(PackageHasDisallowedOrgname, PACKAGE_HAS_DISALLOWED_ORGNAME),
    /** File conflict found when installing the package */
    ERROR_CODE_ENUM_ENTRY(FileConflict, FILE_CONFLICT),
    /** Attempt to install the same package */
    ERROR_CODE_ENUM_ENTRY(SamePackageIsInstalled, SAME_PACKAGE_IS_INSTALLED),
    /** The passed arguments are not valid */
    ERROR_CODE_ENUM_ENTRY(InvalidArguments, INVALID_ARGUMENTS),
    /** The package source not allowed by administrator */
    ERROR_CODE_ENUM_ENTRY(SourceNotAllowed, SOURCE_NOT_ALLOWED),
    /** Package with this unique profile is already installed */
    ERROR_CODE_ENUM_ENTRY(PackageProfileConflict, PACKAGE_PROFILE_CONFLICT),
    /** Package signature could not be verified */
    ERROR_CODE_ENUM_ENTRY(SignatureVerificationFailed, SIGNATURE_VERIFICATION_FAILED),
    /** The package has a developer signature with a disallowed subgroup */
    ERROR_CODE_ENUM_ENTRY(DisallowedDeveloperSubgroup, DISALLOWED_DEVELOPER_SUBGROUP),
};

#ifndef __cplusplus
typedef enum ErrorCode ErrorCode;
#endif

#ifdef __cplusplus
} /* namespace Common */
#endif

#endif /* COMMON_ERROR_CODE_ENUM_HPP */
