// SPDX-FileCopyrightText: 2023 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

#ifndef VIDEOENCODERH264STREAMCAMERA_H
#define VIDEOENCODERH264STREAMCAMERA_H

#include "clock.h"
#include "StreamCameraUtil.h"
#include "StreamCameraVideoBufferNative.h"

#include <streamcamera/streamcamera-codec.h>
#include <streamcamera/streamcamera.h>

#include "api/video_codecs/video_encoder.h"
#include "modules/video_coding/include/video_codec_interface.h"
#include "rtc_base/checks.h"
#include "rtc_base/logging.h"

#include <memory>

class VideoEncoderH264StreamCamera : public webrtc::VideoEncoder
{
public:
    class VideoEncoderListenerImpl : public Aurora::StreamCamera::VideoEncoderListener
    {
    public:
        VideoEncoderListenerImpl(VideoEncoderH264StreamCamera &encoder)
            : encoder_(encoder)
        {}

        ~VideoEncoderListenerImpl() override = default;

        void onEncodedFrame(uint8_t *data,
                            size_t size,
                            uint64_t timestampUs,
                            Aurora::StreamCamera::FrameType frameType) override
        {
            encoder_.onEncodedFrame(data, size, timestampUs, frameType);
        }

        void onEncoderParameterChanged(Aurora::StreamCamera::VideoEncoderParameter,
                                       const std::string &value) override
        {
            RTC_LOG_F(LS_INFO);
        }

        void onEncoderError(const std::string &errorDescription) override
        {
            RTC_LOG_F(LS_ERROR) << errorDescription;
        }

        void onEncoderEOS() override { RTC_LOG_F(LS_INFO); }

        VideoEncoderH264StreamCamera &encoder_;
    };

    static std::unique_ptr<VideoEncoderH264StreamCamera> Create()
    {
        return std::make_unique<VideoEncoderH264StreamCamera>();
    }

    VideoEncoderH264StreamCamera()
        : codec_manager_(*StreamCameraCodecManager())
        , listener_(*this)
    {
        RTC_LOG_F(LS_INFO);
    }

    ~VideoEncoderH264StreamCamera() override
    {
        encoder_.reset();
    }

    /* ABSL_DEPRECATED("bugs.webrtc.org/10720") */
    int32_t InitEncode(const webrtc::VideoCodec *codec_settings,
                       int32_t number_of_cores,
                       size_t max_payload_size) override
    {
        RTC_LOG_F(LS_INFO) << "";
        return WEBRTC_VIDEO_CODEC_ERROR;
    }

    int InitEncode(const webrtc::VideoCodec *codec_settings,
                   const webrtc::VideoEncoder::Settings &settings_unused) override
    {
        if (!codec_manager_.videoEncoderAvailable(Aurora::StreamCamera::CodecType::H264))
            return WEBRTC_VIDEO_CODEC_ERROR;
        if (!(encoder_ = codec_manager_.createVideoEncoder(Aurora::StreamCamera::CodecType::H264)))
            return WEBRTC_VIDEO_CODEC_ERROR;
        pixel_format_list_ = encoder_->getSupportedPixelFormats();
        RTC_LOG_F(LS_INFO) << "pixel_format_list_: "
                           << pix_format_list_str(pixel_format_list_).c_str();

        codec_settings_ = *codec_settings;

        PrintEncoderParamRange();
        encoder_meta_.codecType = Aurora::StreamCamera::CodecType::H264;
        encoder_meta_.width = codec_settings_.width;
        encoder_meta_.height = codec_settings_.height;
        encoder_meta_.stride = codec_settings_.width;
        encoder_meta_.sliceHeight = codec_settings_.height;
        // Codec_settings uses kbits/second; encoder uses bits/second.
        encoder_meta_.bitrate = codec_settings_.startBitrate * 1000;
        encoder_meta_.framerate = codec_settings_.maxFramerate;

        want_fps_ = codec_settings_.maxFramerate;

        RTC_LOG_F(LS_INFO) << " size: " << encoder_meta_.width << "x" << encoder_meta_.height
                           << " bitrate: " << encoder_meta_.bitrate
                           << " framerate: " << encoder_meta_.framerate;

        RTC_DCHECK(encoder_->setParameter(Aurora::StreamCamera::VideoEncoderParameter::BitRateMode,
                                          "constant"));
        encoder_->setParameter(Aurora::StreamCamera::VideoEncoderParameter::PreprocessRotation, std::to_string(rotationAngle_));
        encoder_->setParameter(Aurora::StreamCamera::VideoEncoderParameter::PreprocessMirrorH, "false");
        encoder_->setParameter(Aurora::StreamCamera::VideoEncoderParameter::PreprocessMirrorV, "false");

        if (!encoder_->init(encoder_meta_)) {
            RTC_LOG_F(LS_ERROR) << "failed: encoder_->init()";
            return WEBRTC_VIDEO_CODEC_ERROR;
        }
        encoder_->lockAndSetListener(&listener_);

        return WEBRTC_VIDEO_CODEC_OK;
    }

    int32_t RegisterEncodeCompleteCallback(webrtc::EncodedImageCallback *callback) override
    {
        RTC_LOG_F(LS_INFO) << "";
        encoded_callback_ = callback;
        return WEBRTC_VIDEO_CODEC_OK;
    }

    int32_t Release() override
    {
        RTC_LOG_F(LS_INFO);
        return WEBRTC_VIDEO_CODEC_OK;
    }

    int32_t Encode(const webrtc::VideoFrame &frame,
                   const std::vector<webrtc::VideoFrameType> *frame_types) override
    {
        // WebRTC tells what FPS it wants in SetRates() about 20 times per second.
        // Camera and encoder FPS is set at initialization and can not be changed now.
        // To have FPS that  WebRTC wants frames that come too fast are dropped.
        size_t want_fps_delta_us = 1000000 / want_fps_;
        uint64_t now_us = uptime_us();
        uint64_t delta_us = now_us - last_encode_us_;
        if (delta_us < want_fps_delta_us) {
#ifdef LOG_ENCODE_DROP_FRAME
            RTC_LOG_F(LS_INFO) << "drop frame: want_fps_: " << want_fps_
                               << " want_fps_delta_us: " << want_fps_delta_us
                               << " delta_us: " << delta_us;
#endif // LOG_ENCODE_DROP_FRAME
            return WEBRTC_VIDEO_CODEC_OK;
        }
        last_encode_us_ = now_us;

        bool is_key_frame = false;
        if (frame_types != nullptr && frame_types->size() > 0)
            if ((*frame_types)[0] == webrtc::VideoFrameType::kVideoFrameKey)
                is_key_frame = true;

        if (frame.video_frame_buffer()->type() == webrtc::VideoFrameBuffer::Type::kI420)
            return Encode_webrtc_I420(frame, is_key_frame);

        RTC_DCHECK(frame.video_frame_buffer()->type() == webrtc::VideoFrameBuffer::Type::kNative);
        rtc::scoped_refptr<webrtc::VideoFrameBuffer> buffer_native = frame.video_frame_buffer();
        StreamCameraVideoBufferNative *buffer = static_cast<StreamCameraVideoBufferNative *>(
            buffer_native.get());
        RTC_DCHECK(buffer->ycbcr_ != nullptr);
        if (rotationAngle_ != buffer->encoderRotationAngle_) {
            rotationAngle_  = buffer->encoderRotationAngle_;
            RTC_LOG_F(LS_INFO) << "rotationAngle_ changed: " << rotationAngle_;
            webrtc::VideoEncoder::Settings settings_unused(webrtc::VideoEncoder::Capabilities(false), 0, 0);
            InitEncode(&codec_settings_, settings_unused);
        }
        encoder_->encode(buffer->ycbcr_, is_key_frame);

        return WEBRTC_VIDEO_CODEC_OK;
    }

    void SetRates(const webrtc::VideoEncoder::RateControlParameters &parameters) override
    {
#ifdef LOG_RATE_CONTROL_PARAMS
        RTC_LOG_F(LS_INFO) << " target_bitrate: " << parameters.target_bitrate.get_sum_bps()
                           << " bitrate: " << parameters.bitrate.get_sum_bps()
                           << " framerate_fps: " << parameters.framerate_fps
                           << " bandwidth_allocation: "
                           << parameters.bandwidth_allocation.bps_or(0);
#endif // LOG_RATE_CONTROL_PARAMS
        if (encoder_meta_.codecType != Aurora::StreamCamera::CodecType::H264) {
            RTC_LOG_F(LS_INFO) << "InitEncode() was not called";
            return;
        }
        want_fps_ = parameters.framerate_fps;
        encoder_->setBitrate(parameters.bitrate.get_sum_bps());
    }

    // Input:   - packet_loss_rate  : The packet loss rate (0.0 to 1.0).
    void OnPacketLossRateUpdate(float packet_loss_rate) override
    {
        if (packet_loss_rate > 0.01)
            RTC_LOG_F(LS_INFO) << packet_loss_rate;
    }

    // Inform the encoder when the round trip time changes.
    void OnRttUpdate(int64_t rtt_ms) override {}
    void OnLossNotification(const webrtc::VideoEncoder::LossNotification &loss_notification) override
    {}

    EncoderInfo GetEncoderInfo() const override
    {
        webrtc::VideoEncoder::EncoderInfo info;
        info.supports_native_handle = true;
        info.implementation_name = "StreamCameraH264";
        info.is_hardware_accelerated = true;
        info.has_trusted_rate_controller = true;
        info.supports_simulcast = false;
        return info;
    }

    int32_t Encode_webrtc_I420(const webrtc::VideoFrame &frame, bool is_key_frame)
    {
        auto buffer = static_cast<const webrtc::I420Buffer *>(frame.video_frame_buffer().get());
        return Encode_webrtc_I420(buffer, is_key_frame, frame.timestamp_us());
    }

    int32_t Encode_webrtc_I420(const webrtc::I420Buffer *buffer,
                               bool is_key_frame,
                               uint64_t timestamp_us)
    {
        auto ycbcr = std::make_shared<Aurora::StreamCamera::YCbCrFrame>();
        ycbcr->chromaStep = 1;
        ycbcr->width = buffer->width();
        ycbcr->height = buffer->height();
        ycbcr->yStride = buffer->StrideY();
        ycbcr->cStride = buffer->StrideU();
        ycbcr->y = buffer->DataY();
        ycbcr->cb = buffer->DataU();
        ycbcr->cr = buffer->DataV();
        ycbcr->timestampUs = timestamp_us;
        encoder_->encode(ycbcr, is_key_frame);

        return WEBRTC_VIDEO_CODEC_OK;
    }

    void onEncodedFrame(uint8_t *data,
                        size_t size,
                        uint64_t timestampUs,
                        Aurora::StreamCamera::FrameType frameType)
    {
        if (encoded_callback_ == nullptr) {
            RTC_LOG_F(LS_INFO) << "encoded_callback_ == nullptr";
            return;
        }
        // TODO check warning:
        // (frame_encode_metadata_writer.cc:261): Frame with no encode
        // started time recordings.
        // Encoder may be reordering frames or not preserving RTP timestamps.
        webrtc::EncodedImage encoded_image;
        auto buffer = webrtc::EncodedImageBuffer::Create(data, size);
        encoded_image.SetEncodedData(buffer);
        encoded_image._frameType = webrtc::VideoFrameType::kVideoFrameDelta;
        if (frameType == Aurora::StreamCamera::FrameType::Key)
            encoded_image._frameType = webrtc::VideoFrameType::kVideoFrameKey;
        encoded_image.SetTimestamp(timestampUs / 1000 * 90);

        webrtc::CodecSpecificInfo codec_specific{};
        codec_specific.codecType = webrtc::kVideoCodecH264;
        codec_specific.codecSpecific.H264.packetization_mode
            = webrtc::H264PacketizationMode::NonInterleaved;
        codec_specific.codecSpecific.H264.temporal_idx = webrtc::kNoTemporalIdx;
        codec_specific.codecSpecific.H264.idr_frame = frameType
                                                      == Aurora::StreamCamera::FrameType::Key;
        codec_specific.codecSpecific.H264.base_layer_sync = false;

        webrtc::EncodedImageCallback::Result res
            = encoded_callback_->OnEncodedImage(encoded_image, &codec_specific);
        if (res.error != webrtc::EncodedImageCallback::Result::OK)
            RTC_LOG_F(LS_ERROR) << "failed: OnEncodedImage()";
    }

    void PrintEncoderParamRange()
    {
        using namespace Aurora::StreamCamera;
        RTC_LOG_F(LS_INFO) << "Encoder parameters:"
                           << "\n"
                           << " BitRateMode: "
                           << encoder_->getParameter(VideoEncoderParameter::BitRateMode) << " ("
                           << encoder_->getParameterRange(VideoEncoderParameter::BitRateMode) << ")"
                           << "\n"
                           << " PreprocessRotation: "
                           << encoder_->getParameter(VideoEncoderParameter::PreprocessRotation)
                           << " ("
                           << encoder_->getParameterRange(VideoEncoderParameter::PreprocessRotation)
                           << ")"
                           << "\n"
                           << " PreprocessMirrorH: "
                           << encoder_->getParameter(VideoEncoderParameter::PreprocessMirrorH)
                           << " ("
                           << encoder_->getParameterRange(VideoEncoderParameter::PreprocessMirrorH)
                           << ")"
                           << "\n"
                           << " PreprocessMirrorV: "
                           << encoder_->getParameter(VideoEncoderParameter::PreprocessMirrorV)
                           << " ("
                           << encoder_->getParameterRange(VideoEncoderParameter::PreprocessMirrorV)
                           << ")"
                           << "\n"
                           << " H264Profile: "
                           << encoder_->getParameter(VideoEncoderParameter::H264Profile) << " ("
                           << encoder_->getParameterRange(VideoEncoderParameter::H264Profile) << ")"
                           << "\n"
                           << " H264Level: "
                           << encoder_->getParameter(VideoEncoderParameter::H264Level) << " ("
                           << encoder_->getParameterRange(VideoEncoderParameter::H264Level) << ")"
                           << "\n"
                           << " H264IFrameIntervalSec: "
                           << encoder_->getParameter(VideoEncoderParameter::H264IFrameIntervalSec)
                           << " ("
                           << encoder_->getParameterRange(
                                  VideoEncoderParameter::H264IFrameIntervalSec)
                           << ")"
                           << "\n";
    }

    Aurora::StreamCamera::CodecManager &codec_manager_;
    std::shared_ptr<Aurora::StreamCamera::VideoEncoder> encoder_ = nullptr;
    std::vector<Aurora::StreamCamera::PixelFormat> pixel_format_list_;
    VideoEncoderListenerImpl listener_;
    webrtc::EncodedImageCallback *encoded_callback_ = nullptr;
    webrtc::VideoCodec codec_settings_ = {};
    Aurora::StreamCamera::VideoEncoderMetadata encoder_meta_ = {};
    size_t want_fps_ = 0;
    uint64_t last_encode_us_ = 0;
    int rotationAngle_ = 0;
};

#endif // VIDEOENCODERH264STREAMCAMERA_H
