/*
 *  Copyright (c) 2022 The WebRTC project authors. All Rights Reserved.
 *
 *  Use of this source code is governed by a BSD-style license
 *  that can be found in the LICENSE file in the root of the source
 *  tree. An additional intellectual property rights grant can be found
 *  in the file PATENTS.  All contributing project authors may
 *  be found in the AUTHORS file in the root of the source tree.
 */

#ifndef API_VIDEO_CODECS_VIDEO_ENCODER_FACTORY_TEMPLATE_OPEN_H264_ADAPTER_H_
#define API_VIDEO_CODECS_VIDEO_ENCODER_FACTORY_TEMPLATE_OPEN_H264_ADAPTER_H_

#include <memory>
#include <vector>

#include "modules/video_coding/codecs/h264/include/h264.h"

#ifdef WEBRTC_AURORA
#include "rtc_base/logging.h"
#include "aurora/VideoEncoderFactoryH264StreamCamera.h"
#endif

namespace webrtc {
// TODO(bugs.webrtc.org/13573): When OpenH264 is no longer a conditional build
//                              target remove this #ifdef.
#if defined(WEBRTC_USE_H264)
struct OpenH264EncoderTemplateAdapter {
  static std::vector<SdpVideoFormat> SupportedFormats() {
    return SupportedH264Codecs();
  }

  static std::unique_ptr<VideoEncoder> CreateEncoder(
      const SdpVideoFormat& format) {
    return H264Encoder::Create(cricket::VideoCodec(format));
  }

  static bool IsScalabilityModeSupported(ScalabilityMode scalability_mode) {
    return H264Encoder::SupportsScalabilityMode(scalability_mode);
  }
};

#ifdef WEBRTC_AURORA
struct H264StreamCameraEncoderTemplateAdapter {
  static std::vector<SdpVideoFormat> SupportedFormats() {
    RTC_LOG_F(LS_INFO);
    return instance()->factory_->GetSupportedFormats();
  }

  static std::unique_ptr<VideoEncoder> CreateEncoder(
      const SdpVideoFormat& format) {
    RTC_LOG_F(LS_INFO);
    return instance()->factory_->CreateVideoEncoder(format);
  }

  static bool IsScalabilityModeSupported(ScalabilityMode scalability_mode) {
    RTC_LOG_F(LS_INFO);
    return H264Encoder::SupportsScalabilityMode(scalability_mode); // TODO find out
  }

  static H264StreamCameraEncoderTemplateAdapter* instance() {
      static H264StreamCameraEncoderTemplateAdapter* instance = nullptr;
      if (!instance)
        instance = new H264StreamCameraEncoderTemplateAdapter;
      return instance;
  }

  H264StreamCameraEncoderTemplateAdapter()
  : factory_(new VideoEncoderFactoryH264StreamCamera)
  {
      RTC_LOG_F(LS_INFO);
  }

  std::unique_ptr<VideoEncoderFactoryH264StreamCamera> factory_;
};
#endif  // WEBRTC_AURORA


#endif  // defined(WEBRTC_USE_H264)
}  // namespace webrtc

#endif  // API_VIDEO_CODECS_VIDEO_ENCODER_FACTORY_TEMPLATE_OPEN_H264_ADAPTER_H_
