/*
 * SPDX-FileCopyrightText: Copyright 2023-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include "ieglcontextdelegate.h"

#include <vector>
#include <QEvent>
#include <QMap>
#include <QString>

//! Aurora namespace
namespace Aurora {
//! WebView namespace
namespace WebView {
//! Contains functions related to web engine initialization
namespace WebEngineContext {

/**
    Initialize browser processes context.

    Web engine creates auxiliary processes during work via forking the original
    process several times. You must call that function as early as
    possible to proper initialize the context and to not mix it with an application logic.

    This function returns only for initial process.

    You can use use additional command line arguments, or pass them from your code using the
    additionalArgs parameter. The supported arguments are listed in the table below.

     Condition                         | Explanation
    |----------------------------------|-------------------------------------------------------|
    | ignore-certificate-errors        | A set of public key hashes for which to ignore
    |                                  | certificate-related errors. If the certificate
    |                                  | chain presented by the server is invalid,
    |                                  | and one or more certificates have public key
    |                                  | hashes that match a key from this list, the error
    |                                  | is ignored. The switch value must be a 5241
    |                                  | comma-separated list of Base64-encoded SHA-256 SPKI
    |                                  | Fingerprints (RFC 7469, Section 2.4).                 |
    | disable-web-security             | Don't enforce the same-origin policy; meant for
    |                                  | website testing only. It disables CORS checks, ORB
    |                                  | mechanism and PNA checks.                             |
    | remote-debugging-port            | Enables remote debug over HTTP on the specified port  |
    | remote-allow-origins             | Enables web socket connections from the specified
    |                                  | origins only. '*' allows any origin.                  |
    | disable-accelerated-video-encode | Disables hardware acceleration of video encode, where
    |                                  | available.                                            |
    | default-encoding                 | Sets the encoding used by default. The default value
    |                                  | is "ISO-8859-1".                                      |

    @deprecated Use InitBrowser(int argc, ...) instead
    @param argc number of command line arguments
    @param argv command line arguments
    @param additionalArgs additional arguments to forward to the engine. Optional argument
    @param eglDelegate EGL context platform provider for GPU composition. May be nullptr for QGuiApplication-based applications
*/
[[deprecated("Use InitBrowser(int argc, ...) instead")]]
void StartProcess(
    int argc,
    char *argv[],
    const std::vector<std::string> &additionalArgs = {},
    IEglContextDelegatePtr eglDelegate = nullptr);

/**
    Initialize browser engine to do actual work.

    You can use use additional command line arguments, or pass them from your code using the
    additionalArgs parameter. The supported arguments are listed in the table below.

     Condition                         | Explanation                                        
    |----------------------------------|-------------------------------------------------------|
    | ignore-certificate-errors        | A set of public key hashes for which to ignore
    |                                  | certificate-related errors. If the certificate
    |                                  | chain presented by the server is invalid,
    |                                  | and one or more certificates have public key
    |                                  | hashes that match a key from this list, the error
    |                                  | is ignored. The switch value must be a 5241
    |                                  | comma-separated list of Base64-encoded SHA-256 SPKI
    |                                  | Fingerprints (RFC 7469, Section 2.4).                 |
    | disable-web-security             | Don't enforce the same-origin policy; meant for
    |                                  | website testing only. It disables CORS checks, ORB
    |                                  | mechanism and PNA checks.                             |
    | host-resolver-rules              | Custom mappings for hostnames. Format:
    |                                  | "MAP <hostname_pattern> <ip_address>[:<port>]".
    |                                  | Example: "MAP example.com 127.0.0.1:8080" or
    |                                  | "MAP *.example.com 192.168.1.1". Multiple rules
    |                                  | can be comma-separated.                               |
    | remote-debugging-port            | Enables remote debug over HTTP on the specified port  |
    | remote-allow-origins             | Enables web socket connections from the specified
    |                                  | origins only. '*' allows any origin.                  |
    | disable-accelerated-video-encode | Disables hardware acceleration of video encode, where
    |                                  | available.                                            |
    | default-encoding                 | Sets the encoding used by default. The default value
    |                                  | is "ISO-8859-1".                                      |
    | browser-subprocess-path          | The path to a separate executable that will be
    |                                  | launched for sub-processes. It can be used in
    |                                  | additionalArgs param only.                            |
    | autoplay-policy                  | Overrides default autoplay policy behavior.
    |                                  | Override this argument (set
    |                                  | autoplay-policy=no-user-gesture-required) so that
    |                                  | media with sound can be started automatically from JS
    |                                  | code. By default, media with sound isn't allowed to
    |                                  | start playing automatically                           |

    <params> serves to customize embedding webview in your application. Embedder should not change it
    if they use system webview library. Supported parameters are:
    | browser-extensions-path          | Path to directory with browser JS extensions. If      |
    |                                  | relative, PackageFilesLocation from libauroraapp is   |
    |                                  | used as reference.QEvent                              |
    | disable-system-extensions        | Define to avoid loading system JS extensions. This    |
    |                                  | may disable some webview functionality depending on   |
    |                                  | system version.                                       |
    | resources-path                   | Path to custom error pages and similar data.          |
    | private-libdir                   | Path to webvew plugins. They serve as the glue to     |
    |                                  | other system components and have use for crossplatform|
    |                                  | solutions.QEvent                                      |

    @param params optional webview params
    @param argc number of command line arguments
    @param argv command line arguments
    @param additionalArgs additional arguments to forward to the engine. Optional argument
    @param eglDelegate EGL context platform provider for GPU composition. May be nullptr for QGuiApplication-based applications
*/
void InitBrowser(
    int argc,
    char *argv[],
    const std::vector<std::string> &additionalArgs = {},
    QMap<QString, QString> params = {},
    IEglContextDelegatePtr eglDelegate = nullptr);

/**
    Initialize browser engine to do actual work.

    @deprecated Use InitBrowser(int argc, ...) instead
    @param params optional webview params
*/
[[deprecated("Use InitBrowser(int argc, ...) instead")]]
void InitBrowser(QMap<QString, QString> params = {});

/**
 * @brief terminate the browser engine and the WebView
 * 
 * After completion, WebView posts ReadyToClose QT event.
 *
 */
void Shutdown();

/**
 * @class ReadyToClose
 * @brief Represents a QT event.
 *
 * Defines the ReadyToClose event that the WebView posts after it has shutdown the browser engine.
 * The application working with WebView should shutdown after receiving this event, and it is
 * not recommended to terminate the application before.
 * 
 */
class ReadyToClose : public QEvent
{
public:
    static const QEvent::Type readyToCloseEventType = static_cast<QEvent::Type>(QEvent::User + 100);

    ReadyToClose()
        : QEvent(readyToCloseEventType)
    {}
};

} // namespace WebEngineContext
} // namespace WebView
} // namespace Aurora
