/*
 * SPDX-FileCopyrightText: Copyright 2024-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include <QObject>

#include <memory>

#include "enums/types.h"

class QQmlEngine;
class QJSEngine;

namespace Aurora {
namespace WebView {

class PreferenceManagerPrivate;

/**
 * @class PreferenceManager
 * @brief used to set and get WebView preferences
 *
 * All WebView preferences are conveniently represented as Q_PROPERTIES.
 *
 * @link preferencemanager.qml Usage Example @endlink
 */
class PreferenceManager : public QObject
{
    Q_OBJECT

    /// indicates if PreferenceManager is ready to use.
    ///
    /// if property is false, wait for the "ready" signal.
    /// Otherwise values returned are not guaranteed to be true
    ///
    /// changing a preference until PreferenceManager readiness
    /// will have no effect
    Q_PROPERTY(bool isReady READ isReady NOTIFY ready)

    /// turns off image loading (non base64 images) when set to false.
    ///
    /// setting this property to true makes WebView load images on a webpage without a need to reload.
    ///
    /// setting this property to false prevents new images from being downloaded. However, cached images
    /// will still be displayed.
    Q_PROPERTY(bool imageLoadingEnabled READ imageLoadingEnabled WRITE setImageLoadingEnabled NOTIFY
                   imageLoadingEnabledChanged)
    /// controls Javascript on/off state. Requires a page reload after being set.
    Q_PROPERTY(bool javascriptEnabled READ javascriptEnabled WRITE setJavascriptEnabled NOTIFY
                   javascriptEnabledChanged)

    /// set backend for GOST encryption
    Q_PROPERTY(GostSSLProvider gostsslProvider READ gostsslProvider WRITE setGostsslProvider NOTIFY
                   gostsslProviderChanged)

    /// controls Do Not Track request state
    Q_PROPERTY(bool doNotTrackEnabled READ doNotTrackEnabled WRITE setDoNotTrackEnabled NOTIFY
                   doNotTrackEnabledChanged)

    /// set default color scheme. Website could ignore saved value.
    Q_PROPERTY(ColorVariant::Type colorScheme READ colorScheme WRITE setColorScheme NOTIFY
                   colorSchemeChanged)

    Q_PROPERTY(bool thirdPartyCookiesAllowed READ thirdPartyCookiesAllowed WRITE setThirdPartyCookiesAllowed NOTIFY
                   thirdPartyCookiesAllowedChanged)

public:
    static QSharedPointer<PreferenceManager> GetInstance(
        QQmlEngine *qmlEngine = nullptr, QJSEngine *jsEngine = nullptr);
    virtual ~PreferenceManager();

    bool isReady() const;

    bool imageLoadingEnabled() const;
    void setImageLoadingEnabled(bool enabled);

    bool javascriptEnabled() const;
    void setJavascriptEnabled(bool enabled);

    /**
     * @brief Type of GOST SSL Provider
     */
    enum GostSSLProvider {
        /// Default chromium encryption provider. GOST isn't supported.
        BuiltIn,
        /// Use CryptoPro for GOST encryption
        CryptoPro,
        /// Use OpenSSL for GOST encryption
        OpenSSL,
    };
    Q_ENUM(GostSSLProvider)

    GostSSLProvider gostsslProvider() const;
    void setGostsslProvider(GostSSLProvider provider);

    bool doNotTrackEnabled() const;
    void setDoNotTrackEnabled(bool enabled);

    ColorVariant::Type colorScheme() const;
    void setColorScheme(ColorVariant::Type colorScheme);

    bool thirdPartyCookiesAllowed() const;
    void setThirdPartyCookiesAllowed(bool allow);

    /// get version of WebView (e.g., 1.95)
    Q_INVOKABLE QString getVersion() const;
    /// get version of Chromium Engine (e.g., 126.0.6478.62)
    Q_INVOKABLE QString getEngineVersion() const;

signals:
    void ready();
    void imageLoadingEnabledChanged();
    void javascriptEnabledChanged();
    void gostsslProviderChanged();
    void doNotTrackEnabledChanged();
    void colorSchemeChanged();
    void thirdPartyCookiesAllowedChanged();

private:
    PreferenceManager();

    std::unique_ptr<PreferenceManagerPrivate> m_d;
};

} // namespace WebView
} // namespace Aurora
