/*
 * SPDX-FileCopyrightText: Copyright 2024-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include "browserextension.h"
#include <cstdint>
#include <QObject>

#include <QMouseEvent>

namespace Aurora {
namespace WebView {

enum class TouchEventType { Press, Move, Release, Canceled };

struct TouchEvent
{
    TouchEventType type = TouchEventType::Press;
    int id = 0;
    double x = 0.;
    double y = 0;
};

enum class MouseEventType { Click, Move, Release };

struct MouseEvent
{
public:
    MouseEvent() {};

    MouseEvent(MouseEventType type, int x, int y, Qt::MouseButton button, bool doubleClick)
        : type(type)
        , x(x)
        , y(y)
        , button(button)
        , doubleClick(doubleClick) {};

    MouseEventType type = MouseEventType::Click;
    int x = 0;
    int y = 0;
    Qt::MouseButton button = Qt::NoButton;
    bool doubleClick = false;
};

enum class KeyEventType { Press, Release, Char };

struct KeyEvent
{
    KeyEventType type = KeyEventType::Press;
    std::uint32_t modifiers = 0;
    int domCode = 0;
    std::uint32_t nativeCode = 0;
    char16_t character = 0;
};

struct WheelEvent
{
public:
    WheelEvent() {};

    WheelEvent(qreal realX, qreal realY, qreal realDeltaX, qreal realDeltaY)
        : x(static_cast<int>(realX))
        , y(static_cast<int>(realY))
        , deltaX(static_cast<int>(realDeltaX))
        , deltaY(static_cast<int>(realDeltaY)) {};

    int x = 0;
    int y = 0;
    int deltaX = 0;
    int deltaY = 0;
};

class IInputExtension : public QObject, public Cef::BrowserExtension
{
    Q_OBJECT
public:
    IInputExtension(Cef::Browser &browser)
        : Cef::BrowserExtension(browser)
    {}
    virtual ~IInputExtension() = default;

    virtual void sendTouchEvent(const TouchEvent &event) = 0;
    virtual void sendKeyEvent(const KeyEvent &event) = 0;
    virtual void sendWheelEvent(const WheelEvent &rawEvent) = 0;
    virtual void sendMouseEvent(const MouseEvent &rawEvent) = 0;

    // Use it to replace a text range in an active Webengine input field.
    // Replaces the text starting from the replacementStart coordinate to the replacementEnd,
    // the coordinates are calculated relative to the beginning of the input field.
    virtual void replaceTextInActiveInputField(
        const QString &text, const int replacementStart, const int replacementEnd)
        = 0;
    virtual Qt::CursorShape cursorShape() const = 0;

    // Drag-n-drop.
    virtual void startDragging() = 0;
    virtual void updateDragCursor(bool isCopyDragDrop) = 0;
    virtual void dragTargetDragOver(double x, double y) = 0;
    virtual void dragTargetDrop(double x, double y) = 0;
    virtual void dragSourceEndedAt(double x, double y) = 0;
    virtual void dragSourceSystemDragEnded() = 0;

signals:
    void showKeyboard(Qt::InputMethodHints mode);
    void hideKeyboard();
    void keyboardTextChanged(const QString &text);
    void keyboardCursorPositionChanged(const int cursorPosition);
    void cursorChanged(Qt::CursorShape cursorShape);
    void textSelectionEnableChanged(bool isEnabled);
    void contentDraggingStarted();
};

} // namespace WebView
} // namespace Aurora
