/*
 * SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include <memory>

#include <QList>
#include <QObject>

namespace Aurora {
namespace WebView {

class DialogRequestInternal;

/**
 * @class DialogRequest
 * @brief Represents a file/folder dialog request
 *
 * Provides means to pick a folder/file(s) or cancel the request
 */
class DialogRequest
{
    Q_GADGET

    ///
    /// shows if this request is about picking a folder or a file(s)
    ///
    /// possible values are in Aurora::WebView::Dialogs
    ///
    Q_PROPERTY(qint32 dialogMode READ dialogMode)
    /// MIME types of files requested by the server
    ///
    /// might also include their extensions
    ///
    /// e.g., ["image/*"]
    /// or  ["image/*", ".png", ".jpg", ...]
    ///
    /// Note that file extensions here don't have "*" at the beginning
    /// (".png" instead of "*.png")
    Q_PROPERTY(QStringList filters READ fileFilters)
    /// file extensions requested by the server (e.g., ["*.png", "*.jpg"])
    ///
    /// These file extensions have "*"s to be used with system file pickers
    Q_PROPERTY(QStringList extensions READ fileExtensions)
    /// descriptions of file types to be picked (e.g., "Image Files")
    Q_PROPERTY(QStringList descriptions READ fileDescriptions)
    /// true if this request is valid (do not use otherwise)
    Q_PROPERTY(bool isValid READ isValid)
public:
    /**
     * @brief Constructs a new DialogRequest object
     * This constructor is declared to register
     * this class in QML
     */
    DialogRequest();
    DialogRequest(const DialogRequest &other) = default;
    /**
     * @brief Destroys the DialogRequest object
     */
    virtual ~DialogRequest() = default;

    qint32 dialogMode() const;
    QStringList fileFilters() const;
    QStringList fileExtensions() const;
    QStringList fileDescriptions() const;
    bool isValid() const;

    /// @brief use it to provide files/folders paths to upload
    /// @param filePaths list of absolute paths to files/folders
    Q_INVOKABLE virtual void pickFiles(QStringList filePaths);
    /// @brief use it to cancel the request
    Q_INVOKABLE virtual void cancel();

private:
    DialogRequest(std::shared_ptr<DialogRequestInternal> dialogRequestInternal);

    std::shared_ptr<DialogRequestInternal> m_d;

    friend class DialogRequestInternal;
};

} // namespace WebView
} // namespace Aurora

Q_DECLARE_METATYPE(Aurora::WebView::DialogRequest)
