/*******************************************************************************
 *
 * SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: MIT
 *
 ******************************************************************************/

#ifndef RM_LIB_CONSTANTS_H
#define RM_LIB_CONSTANTS_H

#include <QString>

namespace RuntimeManager {

/**
 * @brief Constants for intent invocation
 * @namespace RuntimeManager::Intents
 *
 * List of predefined system intents. Note that applications are free to define
 * their own intents, with a name starting with the \c "X-" prefix.
 *
 * For more information, see the @ref standard-intents "Standard intents" page.
 *
 * @sa IntentsInvoker::invoke()
 */
namespace Intents {

/**
 * @brief Start the application given in the Hints::PreferredHandler hint.
 *
 * NOTE: Security policies might forbid applications from starting other
 * applications.
 */
extern const QString Start;

/**
 * @brief Ask the system to open the given URI.
 *
 * The URI given in Params::Uri will be opened by invoking a suitable
 * application. An error will be returned if an application capable of
 * displaying the given URI was not found. The Hints::PreferredHandler hint can
 * be used to suggest the application to be used (this is not recommended,
 * since the security policy might block the request).
 */
extern const QString OpenUri;

/**
 * @brief Intent parameters
 * @namespace RuntimeManager::Intents::Params
 *
 * Parameters passed to the intent handler application. Note that this section
 * only lists the parameters supported by the predefined system intents:
 * applications are free to declare custom intents and define their own
 * parameters as free strings.
 *
 * @sa IntentsInvoker::invoke()
 */
namespace Params {

/**
 * @brief: the URI to be opened by the Intents::OpenUri intent.
 */
extern const QString Uri;

} // namespace Params

/**
 * @brief Intent hints
 * @namespace RuntimeManager::Intents::Hints
 *
 * Hints which can be passed to IntentsInvoker::invoke(). These hints affect
 * the behavior of the runtime-manager in respect to processing the intent
 * invocation, such as the selection of the handler application and the files
 * that need to be shared with the intent handler application. Note that hints
 * are not passed to the intent handler (only intent parameters are).
 */
namespace Hints {

/**
 * @brief Hosts to be ignored by OpenURI hint
 *
 * This hint is only valid for the Intents::OpenUri intent. It contains a list
 * of hosts that will be ignored when looking up for an intent handler: it will
 * cause all desktop file rules of the type \c x-url-handler to be ignored.
 * Setting this to \c "all" will cause all host rules to be ignored.
 */
extern const QString IgnoreHosts;

/**
 * @brief Schemes to be ignored by OpenURI hint
 *
 * This hint is only valid for the Intents::OpenUri intent. It contains a list
 * of schemes that will be ignored when looking up for an intent handler: it will
 * cause all desktop file rules of the type \c x-scheme-handler to be ignored.
 * Setting this to \c "all" will cause all scheme rules to be ignored.
 */
extern const QString IgnoreSchemes;

/**
 * @brief the application ID of the desired handler
 *
 * Note: this hint will be honoured only if the two applications come from the
 * same Organization.
 */
extern const QString PreferredHandler;

/**
 * @brief JSON path of files to be shared in read-only mode.
 *
 * This is used to share files with the intent handler in read-only mode: this
 * hint should be a list of strings, each of which is a JSON path indicating
 * which values in the parameters dictionary contain the paths to the files
 * that need to be made available to the target application. For example, if
 * the \c inputFile key in the parameters dictionary contains the path to a
 * file that needs to be accessed by the intent handler in read-only mode, this
 * hint should be set to \c ["inputFile"].
 *
 * Note: this hint is only valid for custom intents.
 */
extern const QString ReadFiles;

/**
 * @brief whether control should return to the caller.
 *
 * A boolean value telling whether the caller should be refocused once the
 * intent handler has completed its task.
 *
 * Note: this hint is only valid for custom intents.
 */
extern const QString RefocusOnReturn;

/**
 * @brief whether any error dialogs should be suppressed.
 *
 * If \c true, the application will take care of informing the user in case an
 * error occurred (for example, if there isn't any suitable handler installed).
 * Otherwise, the system itself will pop up an error message if needed.
 */
extern const QString SuppressErrorDialog;

/**
 * @brief JSON path of files to be shared in read-write mode.
 *
 * This is used to share files with the intent handler in read-write mode: this
 * hint should be a list of strings, each of which is a JSON path indicating
 * which values in the parameters dictionary contain the paths to the files
 * that need to be made available to the target application. For example, if
 * the \c outputFile key in the parameters dictionary contains the path to a
 * file that needs to be accessed by the intent handler in read-write mode, this
 * hint should be set to \c ["outputFile"].
 *
 * Note: this hint is only valid for custom intents.
 */
extern const QString WriteFiles;

} // namespace Hints

} // namespace Intents

} // namespace RuntimeManager

#endif // RM_LIB_CONSTANTS_H
