/*
 * SPDX-FileCopyrightText: Copyright 2023-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

/**
 * @file
 * @brief Contains types like error or other helper codes
 */

#pragma once

#include <QObject>

namespace Aurora {
namespace WebView {

/**
 * @brief Download interrupt reasons
 *
 * This is a wrapper for cef_download_interrupt_reason_t
 */
class DownloadInterruptReason
{
    Q_GADGET
public:
    /**
     * @brief Interrupt reason code
     *
     * \internal
     * Must be kept in sync with cef_types.h -> cef_download_interrupt_reason_t.
     */
    enum Code {
        CEF_DOWNLOAD_INTERRUPT_REASON_NONE = 0,
        CEF_DOWNLOAD_INTERRUPT_REASON_FILE_FAILED = 1,
        CEF_DOWNLOAD_INTERRUPT_REASON_FILE_ACCESS_DENIED = 2,
        CEF_DOWNLOAD_INTERRUPT_REASON_FILE_NO_SPACE = 3,
        CEF_DOWNLOAD_INTERRUPT_REASON_FILE_NAME_TOO_LONG = 5,
        CEF_DOWNLOAD_INTERRUPT_REASON_FILE_TOO_LARGE = 6,
        CEF_DOWNLOAD_INTERRUPT_REASON_FILE_VIRUS_INFECTED = 7,
        CEF_DOWNLOAD_INTERRUPT_REASON_FILE_TRANSIENT_ERROR = 10,
        CEF_DOWNLOAD_INTERRUPT_REASON_FILE_BLOCKED = 11,
        CEF_DOWNLOAD_INTERRUPT_REASON_FILE_SECURITY_CHECK_FAILED = 12,
        CEF_DOWNLOAD_INTERRUPT_REASON_FILE_TOO_SHORT = 13,
        CEF_DOWNLOAD_INTERRUPT_REASON_FILE_HASH_MISMATCH = 14,
        CEF_DOWNLOAD_INTERRUPT_REASON_FILE_SAME_AS_SOURCE = 15,
        CEF_DOWNLOAD_INTERRUPT_REASON_NETWORK_FAILED = 20,
        CEF_DOWNLOAD_INTERRUPT_REASON_NETWORK_TIMEOUT = 21,
        CEF_DOWNLOAD_INTERRUPT_REASON_NETWORK_DISCONNECTED = 22,
        CEF_DOWNLOAD_INTERRUPT_REASON_NETWORK_SERVER_DOWN = 23,
        CEF_DOWNLOAD_INTERRUPT_REASON_NETWORK_INVALID_REQUEST = 24,
        CEF_DOWNLOAD_INTERRUPT_REASON_SERVER_FAILED = 30,
        CEF_DOWNLOAD_INTERRUPT_REASON_SERVER_NO_RANGE = 31,
        CEF_DOWNLOAD_INTERRUPT_REASON_SERVER_BAD_CONTENT = 33,
        CEF_DOWNLOAD_INTERRUPT_REASON_SERVER_UNAUTHORIZED = 34,
        CEF_DOWNLOAD_INTERRUPT_REASON_SERVER_CERT_PROBLEM = 35,
        CEF_DOWNLOAD_INTERRUPT_REASON_SERVER_FORBIDDEN = 36,
        CEF_DOWNLOAD_INTERRUPT_REASON_SERVER_UNREACHABLE = 37,
        CEF_DOWNLOAD_INTERRUPT_REASON_SERVER_CONTENT_LENGTH_MISMATCH = 38,
        CEF_DOWNLOAD_INTERRUPT_REASON_SERVER_CROSS_ORIGIN_REDIRECT = 39,
        CEF_DOWNLOAD_INTERRUPT_REASON_USER_CANCELED = 40,
        CEF_DOWNLOAD_INTERRUPT_REASON_USER_SHUTDOWN = 41,
        CEF_DOWNLOAD_INTERRUPT_REASON_CRASH = 50,
    };

    Q_ENUM(Code)
};

/**
 * @brief contains enums to work with permissions.
 */
class Permissions
{
    Q_GADGET
public:
    /**
     * @brief Request code
     *
     * \internal
     * Must be kept in sync with cef_types.h -> cef_permission_request_types_t.
     */
    enum RequestType {
        CEF_PERMISSION_TYPE_NONE = 0,
        CEF_PERMISSION_TYPE_ACCESSIBILITY_EVENTS = 1 << 0,
        CEF_PERMISSION_TYPE_AR_SESSION = 1 << 1,
        CEF_PERMISSION_TYPE_CAMERA_PAN_TILT_ZOOM = 1 << 2,
        CEF_PERMISSION_TYPE_CAMERA_STREAM = 1 << 3,
        CEF_PERMISSION_TYPE_CAPTURED_SURFACE_CONTROL = 1 << 4,
        CEF_PERMISSION_TYPE_CLIPBOARD = 1 << 5,
        CEF_PERMISSION_TYPE_TOP_LEVEL_STORAGE_ACCESS = 1 << 6,
        CEF_PERMISSION_TYPE_DISK_QUOTA = 1 << 7,
        CEF_PERMISSION_TYPE_LOCAL_FONTS = 1 << 8,
        CEF_PERMISSION_TYPE_GEOLOCATION = 1 << 9,
        CEF_PERMISSION_TYPE_IDENTITY_PROVIDER = 1 << 10,
        CEF_PERMISSION_TYPE_IDLE_DETECTION = 1 << 11,
        CEF_PERMISSION_TYPE_MIC_STREAM = 1 << 12,
        CEF_PERMISSION_TYPE_MIDI_SYSEX = 1 << 13,
        CEF_PERMISSION_TYPE_MULTIPLE_DOWNLOADS = 1 << 14,
        CEF_PERMISSION_TYPE_NOTIFICATIONS = 1 << 15,
        CEF_PERMISSION_TYPE_KEYBOARD_LOCK = 1 << 16,
        CEF_PERMISSION_TYPE_POINTER_LOCK = 1 << 17,
        CEF_PERMISSION_TYPE_PROTECTED_MEDIA_IDENTIFIER = 1 << 18,
        CEF_PERMISSION_TYPE_REGISTER_PROTOCOL_HANDLER = 1 << 19,
        CEF_PERMISSION_TYPE_STORAGE_ACCESS = 1 << 20,
        CEF_PERMISSION_TYPE_VR_SESSION = 1 << 21,
        CEF_PERMISSION_TYPE_WINDOW_MANAGEMENT = 1 << 22,
        CEF_PERMISSION_TYPE_FILE_SYSTEM_ACCESS = 1 << 23,
    };

    /**
     * @brief Media request code
     *
     * \internal
     * Must be kept in sync with cef_types.h -> cef_media_access_permission_types_t.
     */
    enum MediaRequestType {
        CEF_MEDIA_PERMISSION_NONE = 0,
        CEF_MEDIA_PERMISSION_DEVICE_AUDIO_CAPTURE = 1 << 0,
        CEF_MEDIA_PERMISSION_DEVICE_VIDEO_CAPTURE = 1 << 1,
        CEF_MEDIA_PERMISSION_DESKTOP_AUDIO_CAPTURE = 1 << 2,
        CEF_MEDIA_PERMISSION_DESKTOP_VIDEO_CAPTURE = 1 << 3,
    };

    /**
     * @brief Permission request result code
     *
     * \internal
     * Must be kept in sync with cef_types.h -> cef_permission_request_result_t.
     */
    enum Result {
        CEF_PERMISSION_RESULT_ACCEPT,
        CEF_PERMISSION_RESULT_ACCEPT_THIS_TIME,
        CEF_PERMISSION_RESULT_DENY,
        CEF_PERMISSION_RESULT_DISMISS,
        CEF_PERMISSION_RESULT_IGNORE,
    };

    Q_ENUM(RequestType)
    Q_ENUM(MediaRequestType)
    Q_ENUM(Result)
};

/**
 * @brief Specifies what element context menu was invoked on
 */
class ContextMenuTypeFlags
{
    Q_GADGET
public:
    enum Code {
        ///
        /// No node is selected.
        ///
        CM_TYPEFLAG_NONE = 0,
        ///
        /// The top page is selected.
        ///
        CM_TYPEFLAG_PAGE = 1 << 0,
        ///
        /// A subframe page is selected.
        ///
        CM_TYPEFLAG_FRAME = 1 << 1,
        ///
        /// A link is selected.
        ///
        CM_TYPEFLAG_LINK = 1 << 2,
        ///
        /// A media node is selected.
        ///
        CM_TYPEFLAG_MEDIA = 1 << 3,
        ///
        /// There is a textual or mixed selection that is selected.
        ///
        CM_TYPEFLAG_SELECTION = 1 << 4,
        ///
        /// An editable element is selected.
        ///
        CM_TYPEFLAG_EDITABLE = 1 << 5,
    };

    Q_ENUM(Code)
};

/**
 * @brief Media type of selected element for a context menu
 */
class ContextMenuMediaType
{
    Q_GADGET
public:
    enum Code {
        ///
        /// No special node is in context.
        ///
        CM_MEDIATYPE_NONE,
        ///
        /// An image node is selected.
        ///
        CM_MEDIATYPE_IMAGE,
        ///
        /// A video node is selected.
        ///
        CM_MEDIATYPE_VIDEO,
        ///
        /// An audio node is selected.
        ///
        CM_MEDIATYPE_AUDIO,
        ///
        /// An canvas node is selected.
        ///
        CM_MEDIATYPE_CANVAS,
        ///
        /// A file node is selected.
        ///
        CM_MEDIATYPE_FILE,
        ///
        /// A plugin node is selected.
        ///
        CM_MEDIATYPE_PLUGIN,
    };

    Q_ENUM(Code)
};

/**
 * @brief Contains context menu media state flags (e.g., if an element can be saved)
 */
class ContextMenuMediaStateFlags
{
    Q_GADGET
public:
    enum Code {
        CM_MEDIAFLAG_NONE = 0,
        CM_MEDIAFLAG_IN_ERROR = 1 << 0,
        CM_MEDIAFLAG_PAUSED = 1 << 1,
        CM_MEDIAFLAG_MUTED = 1 << 2,
        CM_MEDIAFLAG_LOOP = 1 << 3,
        CM_MEDIAFLAG_CAN_SAVE = 1 << 4,
        CM_MEDIAFLAG_HAS_AUDIO = 1 << 5,
        CM_MEDIAFLAG_CAN_TOGGLE_CONTROLS = 1 << 6,
        CM_MEDIAFLAG_CONTROLS = 1 << 7,
        CM_MEDIAFLAG_CAN_PRINT = 1 << 8,
        CM_MEDIAFLAG_CAN_ROTATE = 1 << 9,
        CM_MEDIAFLAG_CAN_PICTURE_IN_PICTURE = 1 << 10,
        CM_MEDIAFLAG_PICTURE_IN_PICTURE = 1 << 11,
        CM_MEDIAFLAG_CAN_LOOP = 1 << 12,
    };

    Q_ENUM(Code)
};

/**
 * @brief Actions that can be performed on the element (cut, copy, etc.)
 */
class ContextMenuEditStateFlags
{
    Q_GADGET
public:
    enum Code {
        CM_EDITFLAG_NONE = 0,
        CM_EDITFLAG_CAN_UNDO = 1 << 0,
        CM_EDITFLAG_CAN_REDO = 1 << 1,
        CM_EDITFLAG_CAN_CUT = 1 << 2,
        CM_EDITFLAG_CAN_COPY = 1 << 3,
        CM_EDITFLAG_CAN_PASTE = 1 << 4,
        CM_EDITFLAG_CAN_DELETE = 1 << 5,
        CM_EDITFLAG_CAN_SELECT_ALL = 1 << 6,
        CM_EDITFLAG_CAN_TRANSLATE = 1 << 7,
        CM_EDITFLAG_CAN_EDIT_RICHLY = 1 << 8,
    };

    Q_ENUM(Code)
};

/**
 * @brief Wrapper for cef_quick_menu_edit_state_flags_t
 */
class QuickMenuEditStateFlags
{
    Q_GADGET
public:
    /**
     * @brief QuickMenuEditStateFlags code
     */
    enum Code {
        QM_EDITFLAG_NONE = 0,
        QM_EDITFLAG_CAN_ELLIPSIS = 1 << 0,
        QM_EDITFLAG_CAN_CUT = 1 << 1,
        QM_EDITFLAG_CAN_COPY = 1 << 2,
        QM_EDITFLAG_CAN_PASTE = 1 << 3,
    };

    Q_ENUM(Code)
};

/**
 * @brief Wrapper for cef_menu_id_t
 */
class MenuId
{
    Q_GADGET
public:
    /**
     * @brief MenuId code
     */
    enum Code {
        ///@{
        /// Navigation.
        MENU_ID_BACK = 100,
        MENU_ID_FORWARD = 101,
        MENU_ID_RELOAD = 102,
        MENU_ID_RELOAD_NOCACHE = 103,
        MENU_ID_STOPLOAD = 104,
        ///@}

        ///@{
        /// Editing.
        MENU_ID_UNDO = 110,
        MENU_ID_REDO = 111,
        MENU_ID_CUT = 112,
        MENU_ID_COPY = 113,
        MENU_ID_PASTE = 114,
        MENU_ID_DELETE = 115,
        MENU_ID_SELECT_ALL = 116,
        ///@}

        ///@{
        /// Miscellaneous.
        MENU_ID_FIND = 130,
        MENU_ID_PRINT = 131,
        MENU_ID_VIEW_SOURCE = 132,
        ///@}

        ///@{
        /// Spell checking word correction suggestions.
        MENU_ID_SPELLCHECK_SUGGESTION_0 = 200,
        MENU_ID_SPELLCHECK_SUGGESTION_1 = 201,
        MENU_ID_SPELLCHECK_SUGGESTION_2 = 202,
        MENU_ID_SPELLCHECK_SUGGESTION_3 = 203,
        MENU_ID_SPELLCHECK_SUGGESTION_4 = 204,
        MENU_ID_SPELLCHECK_SUGGESTION_LAST = 204,
        MENU_ID_NO_SPELLING_SUGGESTIONS = 205,
        MENU_ID_ADD_TO_DICTIONARY = 206,
        ///@}

        ///@{
        /// Custom menu items originating from the renderer process.
        MENU_ID_CUSTOM_FIRST = 220,
        MENU_ID_CUSTOM_LAST = 250,
        ///@}

        ///@{
        /// All user-defined menu IDs should come between MENU_ID_USER_FIRST and
        /// MENU_ID_USER_LAST to avoid overlapping the Chromium and CEF ID ranges
        /// defined in the tools/gritsettings/resource_ids file.
        MENU_ID_USER_FIRST = 26500,
        MENU_ID_USER_COPY_LINK = 26501,
        MENU_ID_USER_SHARE_LINK = 26502,
        MENU_ID_USER_DOWNLOAD_IMAGE = 26503,
        MENU_ID_USER_COPY_IMAGE = 26504,
        MENU_ID_USER_SHARE = 26505,
        MENU_ID_USER_LAST = 28500,
        ///@}
    };

    Q_ENUM(Code)
};

/**
 * @brief Wrapper for cef_horizontal_alignment_t
 */
class HorizontalAlignment
{
    Q_GADGET
public:
    /**
     * @brief HorizontalAlignment type
     */
    enum Type {
        /// Align the text's left edge with that of its display area.
        CEF_HORIZONTAL_ALIGNMENT_LEFT,
        /// Align the text's center with that of its display area.
        CEF_HORIZONTAL_ALIGNMENT_CENTER,
        /// Align the text's right edge with that of its display area.
        CEF_HORIZONTAL_ALIGNMENT_RIGHT,
    };
    Q_ENUM(Type)
};

/**
 * @brief Contains enums for dialog handling
 */
class Dialogs
{
    Q_GADGET
public:
    enum Mode {
        /// Requires that the file exists before allowing the user to pick it.
        FILE_DIALOG_OPEN = 0,
        /// Like Open, but allows picking multiple files to open.
        FILE_DIALOG_OPEN_MULTIPLE,
        /// Like Open, but selects a folder to open.
        FILE_DIALOG_OPEN_FOLDER,
        /// Allows picking a nonexistent file, and prompts to overwrite if the file
        /// already exists.
        FILE_DIALOG_SAVE,
    };

    Q_ENUM(Mode)
};

/**
 * @brief Represents current device mode
 */
class DeviceMode
{
    Q_GADGET
public:
    enum FormFactor {
        /// Mobile mode
        MOBILE,
        /// Desktop mode
        DESKTOP,
    };

    Q_ENUM(FormFactor)
};

/**
 * @brief Contains enums that represent certificate fields
 */
class Certificates
{
    Q_GADGET
public:
    enum CertificateInfoTypeKnown {
        CommonName,         ///< The common name (eg person), id = "2.5.4.3"
        Email,              ///< Email address, id = "GeneralName.rfc822Name"
        EmailLegacy,        ///< PKCS#9 Email field, id = "1.2.840.113549.1.9.1"
        Organization,       ///< An organisation (eg company), id = "2.5.4.10"
        OrganizationalUnit, ///< An part of an organisation (eg a division or branch), id = "2.5.4.11"
        Locality,           ///< The locality (eg city, a shire, or part of a state), id = "2.5.4.7"
        IncorporationLocality, ///< The locality of incorporation (EV certificates), id = "1.3.6.1.4.1.311.60.2.1.1"
        State,                 ///< The state within the country, id = "2.5.4.8"
        IncorporationState, ///< The state of incorporation (EV certificates), id = "1.3.6.1.4.1.311.60.2.1.2"
        Country,            ///< The country, id = "2.5.4.6"
        IncorporationCountry, ///< The country of incorporation (EV certificates), id = "1.3.6.1.4.1.311.60.2.1.3"
        URI,       ///< Uniform Resource Identifier, id = "GeneralName.uniformResourceIdentifier"
        DNS,       ///< DNS name, id = "GeneralName.dNSName"
        IPAddress, ///< IP address, id = "GeneralName.iPAddress"
        XMPP,  ///< XMPP address (see http://www.ietf.org/rfc/rfc3920.txt), id = "1.3.6.1.5.5.7.8.5"
        INN,   ///< Taxpayer Identification Number, id = "1.2.643.3.131.1.1"
        OGRN,  ///< Main state registration number, id = "1.2.643.100.1"
        SNILS, ///< Insurance number of an individual personal account, id = "1.2.643.100.3"
        INNLE, ///< INN for Legal Entity, id = "1.2.643.100.4"
        OGRNIP, ///< OGRN for individual entrepreneur, id = "1.2.643.100.5"
        IdentificationKind ///< How was the certificate owner identified when issuing a certificate, id = "1.2.643.100.114"
    };

    enum CertificateSection {
        DN,             ///< Distinguished name (the primary name)
        AlternativeName ///< Alternative name
    };

    /**
     * @brief Certificate status
     *
     * Wrapper for cef_cert_status_t
     */
    enum CertificateStatus {
        CERT_STATUS_NONE = 0,
        ///@{
        /// Error statuses
        CERT_STATUS_COMMON_NAME_INVALID = 1 << 0,
        CERT_STATUS_DATE_INVALID = 1 << 1,
        CERT_STATUS_AUTHORITY_INVALID = 1 << 2,
        // 1 << 3 is reserved for ERR_CERT_CONTAINS_ERRORS (not useful with WinHTTP).
        CERT_STATUS_NO_REVOCATION_MECHANISM = 1 << 4,
        CERT_STATUS_UNABLE_TO_CHECK_REVOCATION = 1 << 5,
        CERT_STATUS_REVOKED = 1 << 6,
        CERT_STATUS_INVALID = 1 << 7,
        CERT_STATUS_WEAK_SIGNATURE_ALGORITHM = 1 << 8,
        // 1 << 9 was used for CERT_STATUS_NOT_IN_DNS
        CERT_STATUS_NON_UNIQUE_NAME = 1 << 10,
        CERT_STATUS_WEAK_KEY = 1 << 11,
        // 1 << 12 was used for CERT_STATUS_WEAK_DH_KEY
        CERT_STATUS_PINNED_KEY_MISSING = 1 << 13,
        CERT_STATUS_NAME_CONSTRAINT_VIOLATION = 1 << 14,
        CERT_STATUS_VALIDITY_TOO_LONG = 1 << 15,
        ///@}

        ///@{
        /// Non-error statuses
        // Bits 16 to 31 are for non-error statuses.
        CERT_STATUS_IS_EV = 1 << 16,
        CERT_STATUS_REV_CHECKING_ENABLED = 1 << 17,
        // Bit 18 was CERT_STATUS_IS_DNSSEC
        CERT_STATUS_SHA1_SIGNATURE_PRESENT = 1 << 19,
        CERT_STATUS_CT_COMPLIANCE_FAILED = 1 << 20,
        ///@}
    };

    Q_ENUM(CertificateInfoTypeKnown)
    Q_ENUM(CertificateSection)
    Q_ENUM(CertificateStatus)
};

/**
 * @brief Provides connection security status information.
 */
class Connection
{
    Q_GADGET
public:
    enum SecurityType {
        /// http or when the certificate is invalid
        Insecure,
        /// https
        Secure,
        /// https with GOST encryption
        SecureGOST
    };

    Q_ENUM(SecurityType)
};

/**
 * @brief Provides print to PDF file margin type.
 */
class PdfPrintMarginType
{
    Q_GADGET
public:
    enum MarginType {
        /// default margin
        PDF_PRINT_MARGIN_DEFAULT,
        /// none margin
        PDF_PRINT_MARGIN_NONE,
        /// custom margin
        PDF_PRINT_MARGIN_CUSTOM
    };

    Q_ENUM(MarginType)
};

/**
 * @brief Provides Cookie properties.
 */
class Cookies
{
    Q_GADGET
public:
    /**
     * @brief Cookie samesite status
     *
     * Wrapper for cef_cookie_same_site_t
     */
    enum CookieSameSite {
        CEF_COOKIE_SAME_SITE_UNSPECIFIED,
        CEF_COOKIE_SAME_SITE_NO_RESTRICTION,
        CEF_COOKIE_SAME_SITE_LAX_MODE,
        CEF_COOKIE_SAME_SITE_STRICT_MODE,
    };

    /**
     * @brief Cookie priority status
     *
     * Wrapper for cef_cookie_priority_t
     */
    enum CookiePriority {
        CEF_COOKIE_PRIORITY_LOW = -1,
        CEF_COOKIE_PRIORITY_MEDIUM = 0,
        CEF_COOKIE_PRIORITY_HIGH = 1,
    };

    Q_ENUM(CookieSameSite)
    Q_ENUM(CookiePriority)
};

/**
 * @brief Wrapper for cef_color_variant_t
 */
class ColorVariant
{
    Q_GADGET
public:
    enum Type {
        CEF_COLOR_VARIANT_SYSTEM,
        CEF_COLOR_VARIANT_LIGHT,
        CEF_COLOR_VARIANT_DARK,
        CEF_COLOR_VARIANT_TONAL_SPOT,
        CEF_COLOR_VARIANT_NEUTRAL,
        CEF_COLOR_VARIANT_VIBRANT,
        CEF_COLOR_VARIANT_EXPRESSIVE,
    };

    Q_ENUM(Type)
};

/**
 * @brief Provides Overscroll modes
 */
class OverscrollMode
{
    Q_GADGET
public:
    enum Type {
        ///
        /// There is no overscroll.
        ///
        OVERSCROLL_MODE_NONE,
        ///
        /// Overscroll mode when the page is scrolled to the bottom and performing a gesture from bottom to top.
        ///
        OVERSCROLL_MODE_NORTH,
        ///
        /// Overscroll mode when the page is scrolled to the top and performing a gesture from top to bottom.
        ///
        OVERSCROLL_MODE_SOUTH,
        ///
        /// Overscroll mode when the page is scrolled to the right and performing a gesture from right to left.
        ///
        OVERSCROLL_MODE_WEST,
        ///
        /// Overscroll mode when the page is scrolled to the left and performing a gesture from left to right.
        ///
        OVERSCROLL_MODE_EAST,
    };

    Q_ENUM(Type)
};

} // namespace WebView
} // namespace Aurora
