/*
 * SPDX-FileCopyrightText: Copyright 2023-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include "iclient.h"
#include "ipermissionhandler.h"

class QQmlEngine;
class QJSEngine;

namespace Aurora {
namespace WebView {

/**
 * @class PermissionHandler
 * @brief Use this singleton to handle web permission requests
 * 
 * @link handlers.qml Usage Example @endlink
 */
class PermissionHandler : public QObject, public IPermissionHandler
{
    Q_OBJECT
public:
    /// @brief default destructor
    virtual ~PermissionHandler() = default;

    /// @brief creates (if doesn't exist) and returns an instance of PermissionHandler
    /// @return PermissionHandler instance
    static QSharedPointer<PermissionHandler> GetInstance(
        QQmlEngine *qmlEngine = nullptr, QJSEngine *jsEngine = nullptr);

    void requestPermissions(const PermissionRequest &permissionRequest) override;
    void requestMediaAccess(const MediaAccessRequest &mediaAccessRequest) override;
    void notifyPermissionRequestDismissed(
        uint64_t promptId, Permissions::Result requestResult) override;
signals:
    /// emitted when permissions are requested (e.g., geolocation).
    /// (see Aurora::WebView::Permissions::RequestType)
    void permissionsRequested(PermissionRequest permissionRequest);
    /// emitted when media access is requested (e.g., camera or microphone).
    /// (see Aurora::WebView::Permissions::MediaRequestType)
    void mediaAccessRequested(MediaAccessRequest mediaAccessRequest);
    /// emitted when a permission requested is dismissed
    void permissionRequestDismissed(quint64 promptId, quint32 requestResult);

private:
    explicit PermissionHandler(IClient *client);
};

} // namespace WebView
} // namespace Aurora
