/*
 * SPDX-FileCopyrightText: Copyright 2024-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include <memory>

#include <QObject>

#include "jsdialogrequest.h"

namespace Aurora {
namespace WebView {

class PromptDialogRequestInternal;

/**
 * @class PromptDialogRequest
 * @brief Represents JavaScript alert dialog request
 * 
 * Use it to process JS alert dialog
 */
class PromptDialogRequest : public JSDialogRequest
{
    Q_GADGET

    /// URL of the web page that called the function
    Q_PROPERTY(QString sourceUrl READ sourceUrl)
    /// message from the function that triggered the dialog
    Q_PROPERTY(QString message READ message)
    /// returns true if the request is valid, do not use otherwise
    Q_PROPERTY(bool isValid READ isValid)
    /// dialog default prompt text, may be present for the prompt function
    Q_PROPERTY(QString defaultInputPlaceholder READ defaultInputPlaceholder)
public:
    /**
     * @brief Constructs a new PromptDialogRequest object
     * This constructor is declared to register
     * this class in QML
     */
    PromptDialogRequest();
    PromptDialogRequest(const PromptDialogRequest &other) = default;
    /**
     * @brief Destroys the PromptDialogRequest object
     */
    ~PromptDialogRequest() = default;

    QString defaultInputPlaceholder() const;

    /**
     * @brief Used to send true to prompt JavaScript function and return to the web page
     * 
     * @param userInput sends a message to prompt JavaScript function from the user
     */
    Q_INVOKABLE virtual void submit(const QString &userInput);
    /**
     * @brief Used to send false to prompt JavaScript function and return to the web page
     */
    Q_INVOKABLE virtual void cancel();

    bool isValid() const override;

private:
    PromptDialogRequest(
        QString originUrl,
        QString messageText,
        std::shared_ptr<PromptDialogRequestInternal> dialogRequestInternal);

    std::shared_ptr<PromptDialogRequestInternal> m_d;

    friend class PromptDialogRequestInternal;
};

} // namespace WebView
} // namespace Aurora

Q_DECLARE_METATYPE(Aurora::WebView::PromptDialogRequest)
