/*
 * SPDX-FileCopyrightText: Copyright 2023-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include <memory>

#include <QObject>
#include <QString>

namespace Aurora {
namespace WebView {

class IClient;

/**
 * @class TabInfo
 * @brief New tab request details.
 *
 */
class TabInfo
{
    Q_GADGET

    /**
     * @brief Url to load.
     */
    Q_PROPERTY(QString url READ url)
    /**
     * @brief Returns true when a tab is requested via an explicit user gesture (e.g.
     * clicking a link), and false when a tab is requested automatically (e.g. via the
     * DomContentLoaded event)
     */
    Q_PROPERTY(bool isUserGesture READ isUserGesture)
    /**
     * @brief Returns true if the tab must be opened in background, and false if
     * it must be opened in foreground
     */
    Q_PROPERTY(bool isBackground READ isBackground)
    /**
     * @brief Returns the id of the source WebView.
     */
    Q_PROPERTY(int webViewId READ webViewId)
public:
    TabInfo(int webViewId, const QString &url, bool isUserGesture, bool isBackground);
    TabInfo();
    virtual ~TabInfo();

    int webViewId() const;
    QString url() const;
    bool isUserGesture() const;
    bool isBackground() const;

private:
    int m_webViewId;
    QString m_url;
    bool m_isUserGesture;
    bool m_isBackground;
};

/**
 * @class PopupInfo
 * @brief Popup request details.
 *
 */
class PopupInfo
{
    Q_GADGET

    /**
     * @brief Url to load.
     */
    Q_PROPERTY(QString url READ url)
    /**
     * @brief Returns true if the popup was requested via an explicit user gesture (e.g.
     * clicking a link), and false if the popup was requested automatically (e.g. via the
     * DomContentLoaded event)
     */
    Q_PROPERTY(bool isUserGesture READ isUserGesture)
    /**
     * @brief Returns the id of the source WebView.
     */
    Q_PROPERTY(int webViewId READ webViewId)
public:
    PopupInfo(int webViewId, const QString &url, bool isUserGesture);
    PopupInfo(); // This constructor is not used, but it is declared
                 // to make passing this object to QML possible
    virtual ~PopupInfo();

    int webViewId() const;
    QString url() const;
    bool isUserGesture() const;

private:
    int m_webViewId;
    QString m_url;
    bool m_isUserGesture;
};

/**
 * @class IPopupHandler
 * @brief Interface of PopupHandler.
 *
 */
class IPopupHandler
{
public:
    /**
     * @brief Registers IPopupHandler implementations in IClient
     * 
     * @param client can be nullptr, default one will be used in that case.
     */
    IPopupHandler(IClient *client = nullptr);

    /**
     * @brief Destroy the IPopupHandler object
     */
    virtual ~IPopupHandler() = default;

    /**
     * @brief this method is called on each tab request.
     *
     * @param info contains additional information about the request.
     */
    virtual void notifyTabRequested(const TabInfo &info) = 0;
    /**
     * @brief this method is called on each popup request.
     *
     * @param info contains additional information about the request.
     */
    virtual void notifyPopupRequested(const PopupInfo &info) = 0;
};

} // namespace WebView
} // namespace Aurora

Q_DECLARE_METATYPE(Aurora::WebView::TabInfo)
Q_DECLARE_METATYPE(Aurora::WebView::PopupInfo)
