/*
 * SPDX-FileCopyrightText: Copyright 2023-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include "enums/types.h"
#include "mediaaccessrequest.h"
#include "permissionrequest.h"

namespace Aurora {
namespace WebView {

class IClient;

/**
 * @class IPermissionHandler
 * @brief You can implement this interface to override the default PermissionHandler.
 *
 * Browser engine starts using an external implementation as soon as it is created.
 * Please note that only the last created implementation is used. As soon as
 * a new implementation is instantiated, all previous implementations become invalid.
 */
class IPermissionHandler
{
public:
    /// @brief Registers IPermissionHandler implementations in IClient
    /// @param client can be nullptr, default one will be used in that case.
    IPermissionHandler(IClient *client = nullptr);

    /// @brief default destructor
    virtual ~IPermissionHandler() = default;

    /// @brief called when permissions are requested (e.g., geolocation).
    ///        (see Aurora::WebView::Permissions::RequestType)
    /// @param permissionRequest
    virtual void requestPermissions(const PermissionRequest &permissionRequest) = 0;

    /// @brief called when media access is requested (e.g., camera or microphone).
    ///        (see Aurora::WebView::Permissions::MediaRequestType)
    /// @param permissionRequest
    virtual void requestMediaAccess(const MediaAccessRequest &mediaAccessRequest) = 0;

    /// called when a permission requested is dismissed
    virtual void notifyPermissionRequestDismissed(
        uint64_t promptId, Permissions::Result requestResult)
        = 0;
};

} // namespace WebView
} // namespace Aurora
