/*
 * SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include "enums/types.h"

#include "certificateinfo.h"

#include <QObject>
#include <QtCrypto/qca_cert.h>

namespace Aurora {
namespace WebView {

/**
 * @class Certificate
 * @brief Represents a X.509 certificate.
 */
class Certificate : public QCA::Certificate
{
    Q_GADGET

    /// Returns the local id of the X.509 certificate.
    Q_PROPERTY(qint32 id READ id)

    /// Date when certificate becomes valid
    Q_PROPERTY(QDateTime validBegin READ notValidBefore)
    /// Date when certificate becomes invalid
    Q_PROPERTY(QDateTime validEnd READ notValidAfter)

    /// Subject info of the certificate
    Q_PROPERTY(Aurora::WebView::CertificateInfo subjectInfo READ subjectInfo)
    /// Issuer info of the certificate
    Q_PROPERTY(Aurora::WebView::CertificateInfo issuerInfo READ issuerInfo)

    /// Certificate common name (CN)
    Q_PROPERTY(QString commonName READ commonName)
    /// Certificate serial number
    Q_PROPERTY(QString serialNumber READ serialNumber)

    /// Returns true when the certificate is empty
    Q_PROPERTY(bool isNull READ isNull)
    /// Returns true when the certificate is valid as a Certificate Authority
    Q_PROPERTY(bool isCA READ isCA)
    /// Returns true if the certificate is self-signed
    Q_PROPERTY(bool isSelfSigned READ isSelfSigned)
public:
    /**
     * @brief Constructs an empty Certificate object
     */
    Certificate();
    /**
     * @brief Constructs a new Certificate object from an existing QCA::Certificate
     *
     * @param id certificate ID.
     * @param certificate certificate to copy info from
     */
    Certificate(qint32 id, const QCA::Certificate &certificate);

    /**
     * @brief Copy constructor
     */
    Certificate(const Certificate &other) = default;

    qint32 id() const;
    QDateTime notValidBefore() const;
    QDateTime notValidAfter() const;
    CertificateInfo subjectInfo() const;
    CertificateInfo issuerInfo() const;
    QString commonName() const;
    QString serialNumber() const;
    bool isCA() const;
    bool isSelfSigned() const;

private:
    quint32 m_id;

    CertificateInfo m_subjectInfoQML;
    CertificateInfo m_issuerInfoQML;
};

} // namespace WebView
} // namespace Aurora

Q_DECLARE_METATYPE(Aurora::WebView::Certificate)
