/*
 * Copyright (c) 2023. Open Mobile Platform LLC.
 *
 * License: MIT
*/

#ifndef G_APP_MANIFEST_H
#define G_APP_MANIFEST_H

#include <glib.h>
#include <stdbool.h>

G_BEGIN_DECLS

/**
 * \file gappmanifest.h
 */

typedef struct _GManifestEntry GManifestEntry;

GManifestEntry *g_manifest_entry_new_from_filename(const char *path);
GManifestEntry *g_manifest_entry_new_from_keyfile(GKeyFile *key_file);
GManifestEntry *g_manifest_entry_new_from_bytes(const char *bytes, gsize bytes_count);
GManifestEntry *g_manifest_entry_ref(GManifestEntry *entry);
void g_manifest_entry_unref(GManifestEntry *entry);
G_DEFINE_AUTOPTR_CLEANUP_FUNC(GManifestEntry, g_manifest_entry_unref)

/**
 * Returns the value of start group name of desktop entry file.
 */
char *g_manifest_entry_get_start_group(GManifestEntry *entry);

/**
 * Returns the value of Type key or NULL if it is
 * not defined in the input desktop entry file.
 */
char *g_manifest_entry_get_app_type(GManifestEntry *entry);

/**
 * Returns the value of Version key or NULL if it is
 * not defined in the input desktop entry file.
 */
char *g_manifest_entry_get_version(GManifestEntry *entry);

/**
 * Returns the unlocalized value of Name key or NULL if it is
 * not defined in the input desktop entry file.
 */
char *g_manifest_entry_get_name_unlocalized(GManifestEntry *entry);

/**
 * Returns the value of GenericName key or NULL if it is
 * not defined in the input desktop entry file.
 */
char *g_manifest_entry_get_generic_name(GManifestEntry *entry);

/**
 * Indicates whether value of NoDisplay key is true or false.
 * Returns NULL if NoDisplay key is undefined.
 */
bool g_manifest_entry_get_no_display(GManifestEntry *entry);

/**
 * Returns the value of Comment key or NULL if it is
 * not defined in the input desktop entry file.
 */
char *g_manifest_entry_get_comment(GManifestEntry *entry);

/**
 * Returns the value of Icon key or NULL if it is
 * not defined in the input desktop entry file.
 */
char *g_manifest_entry_get_icon(GManifestEntry *entry);

/**
 * Indicates whether value of Hidden key is true or false.
 * Returns NULL if Hidden key is undefined.
 */
bool g_manifest_entry_get_hidden(GManifestEntry *entry);

/**
 * Returns the value of OnlyShowIn key or NULL if it is
 * not defined in the input desktop entry file.
 */
GStrv g_manifest_entry_get_only_show_in(GManifestEntry *entry, gsize *length);

/**
 * Returns the value of NotShowIn key or NULL if it is
 * not defined in the input desktop entry file.
 */
GStrv g_manifest_entry_get_not_show_in(GManifestEntry *entry, gsize *length);

/**
 * Returns the value of TryExec key or NULL if it is
 * not defined in the input desktop entry file.
 */
char *g_manifest_entry_get_try_exec(GManifestEntry *entry);

/**
 * Returns the value of Exec key or NULL if it is
 * not defined in the input desktop entry file.
 */
char *g_manifest_entry_get_exec(GManifestEntry *entry);

/**
 * Returns the value of Path key or NULL if it is
 * not defined in the input desktop entry file.
 */
char *g_manifest_entry_get_path(GManifestEntry *entry);

/**
 * Indicates whether value of Terminal key is true or false.
 * Returns NULL if Terminal key is undefined.
 */
bool g_manifest_entry_get_terminal(GManifestEntry *entry);

/**
 * Returns the value of MimeTypes key or NULL if it is
 * not defined in the input desktop entry file.
 */
GStrv g_manifest_entry_get_mime_type(GManifestEntry *entry, gsize *length);

/**
 * Returns the value of Categories key or NULL if it is
 * not defined in the input desktop entry file.
 */
GStrv g_manifest_entry_get_categories(GManifestEntry *entry, gsize *length);

/**
 * Indicates whether value of StartupNotify key is true or false.
 * Returns NULL if StartupNotify key is undefined.
 */
bool g_manifest_entry_get_startup_notify(GManifestEntry *entry);

/**
 * Returns the value of StartupWMClass key or NULL if it is
 * not defined in the input desktop entry file.
 */
char *g_manifest_entry_get_startup_wm_class(GManifestEntry *entry);

/**
 * Returns the value of URL key or NULL if it is
 * not defined in the input desktop entry file.
 */
char *g_manifest_entry_get_url(GManifestEntry *entry);

/**
 * Returns the value of X-Osso-Service key or NULL if it is
 * not defined in the input desktop entry file.
 */
char *g_manifest_entry_get_x_maemo_service(GManifestEntry *entry);

/**
 * Returns the value of X-Service-Exec key or NULL if it is
 * not defined in the input desktop entry file.
 */
char *g_manifest_entry_get_service_exec(GManifestEntry *entry);

/* Generic API */

/**
 * Returns the value associated with key under group_name translated
 * in the given locale if available. If locale is NULL then the current
 * locale is assumed.
 */
char *g_manifest_entry_get_locale_string(GManifestEntry *entry, const char *group, const char *key);

/**
 * Returns the value of the group-key stored as "group/key"
 * key or NULL if it is not defined in the input desktop entry file.
 */
char *g_manifest_entry_value(GManifestEntry *entry, const char *group, const char *key);

/**
 * Returns the value of the group-key stored as "group/key"
 * key or NULL if it is not defined in the input desktop entry file.
 */
GStrv g_manifest_entry_strv_value(GManifestEntry *entry,
                                  const char *group,
                                  const char *key,
                                  gsize *length);

/**
 * Returns the value of the group-key stored as "group/key"
 * key or NULL if it is not defined in the input desktop entry file.
 */
bool g_manifest_entry_bool_value(GManifestEntry *entry, const char *group, const char *key);

/**
 * Indicates whether desktop file contains group/key or not.
 * Returns NULL if key is not present.
 */
bool g_manifest_entry_contains(GManifestEntry *entry, const char *group, const char *key);

/**
 * Indicates whether desktop file contains group or not.
 * Returns NULL if group is not present.
 */
bool g_manifest_entry_contains_group(GManifestEntry *entry, const char *group);

/**
 * Returns all keys defined in specific group or NULL
 * if there is no such group.
 */
GStrv g_manifest_entry_group_keys(GManifestEntry *entry, const char *group, gsize *length);

/**
 * Returns all groups defined in keyfile or NULL if there are no groups.
 */
GStrv g_manifest_entry_groups(GManifestEntry *entry, gsize *length);

/* Sandboxing-related API */

/**
 * Returns the application name used for sandboxing or NULL if it is
 * not defined in the input desktop entry file.
 */
char *g_manifest_entry_get_app_name(GManifestEntry *entry);

/**
 * Returns the organization name used for sandboxing or NULL if it is
 * not defined in the input desktop entry file.
 */
char *g_manifest_entry_get_app_orgname(GManifestEntry *entry);

/**
 * Returns the application permissions used for sandboxing or NULL
 * if it is not defined in the input desktop entry file.
 */
GStrv g_manifest_entry_get_app_permissions(GManifestEntry *entry, gsize *length);

/**
 * Return the value of two fields OrganizationName and 
 * ApplicationName in such form: <orgName>.<appName>  or NULL if 
 * one of them is not defined in the input desktop entry file.
*/
char *g_manifest_entry_get_application_id(GManifestEntry *entry);

/**
 * Returns the value of X-Maemo-Object-Path key or NULL if it is
 * not defined in the input desktop entry file.
 */
char *g_manifest_entry_x_maemo_object_path(GManifestEntry *entry);

/**
 * Returns the value of X-Maemo-Method key or NULL if it is
 * not defined in the input desktop entry file.
 */
char *g_manifest_entry_get_x_maemo_method(GManifestEntry *entry);

/**
 * Returns the value of X-Nemo-Signle-Instance key or NULL if it is
 * not defined in the input desktop entry file.
 */
char *g_manifest_entry_get_x_nemo_single_instance(GManifestEntry *entry);

/**
 * Returns the value of Sandboxing key or NULL if it is
 * not defined in the input desktop entry file.
 */
char *g_manifest_entry_get_sandboxing(GManifestEntry *entry);

/**
 * Returns the value of execDBus key or NULL if it is
 * not defined in the input desktop entry file.
 */
char *g_manifest_entry_get_exec_dbus(GManifestEntry *entry);

G_END_DECLS

#endif // G_APP_MANIFEST_H
