/*******************************************************************************
 *
 * SPDX-FileCopyrightText: 2023-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: MIT
 *
 ******************************************************************************/

#ifndef RM_LIB_INTENTS_INVOKER_H
#define RM_LIB_INTENTS_INVOKER_H

#include "error.h"
#include "global.h"

#include <QObject>
#include <QScopedPointer>
#include <QUrl>

class QJsonObject;
namespace RuntimeManager {

class IntentsInvokerPrivate;
/**
 * @brief Intent invoker.
 * @class IntentsInvoker intents_invoker.h <RuntimeManager/IntentsInvoker>
 *
 * Allow applications invoke actions on other applications.
 */
class RM_EXPORT IntentsInvoker: public QObject
{
    Q_OBJECT
public:
    /**
     * Construct an intent invoker. Note that this object must not be destroyed
     * before the replyReceived() signal is emitted, or a reply will not be
     * received.
     */
    IntentsInvoker(QObject *parent = Q_NULLPTR);
    ~IntentsInvoker();

    /**
     * @brief Invoke specified intent.
     *
     * The intents defined by the OS are listed in the @ref standard-intents
     * "Standard intents" page. This method can also be used to invoke
     * non-standard intents (make sure that the intent name starts with an "X-"
     * prefix.
     *
     * Since intents need some time to processed, the reply will be delivered
     * asynchronously by replyReceived() signal.
     *
     * @param intentName The name of the intent to invoke
     * @param hints Hints to the RuntimeManager service; for example, the
     *        "preferredHandler" hint can be used to specify the application ID
     *        of the desired handler (this will be honoured only if the two
     *        applications come from the same Organization).
     * @param params The parameters for the intent
     */
    void invoke(const QString &intentName,
                const QJsonObject &hints,
                const QJsonObject &params);

    /**
     * @brief Convert an intent to a URI
     *
     * @param intentName The name of the intent to invoke
     * @param hints Hints to the RuntimeManager service; see the invoke()
     *        method for details
     * @param params The parameters for the intent
     */
    static QUrl intentToUri(const QString &intentName,
                            const QJsonObject &hints,
                            const QJsonObject &params);

Q_SIGNALS:
    /**
     * @brief Completion callback
     *
     * This signal will be emitted when the reply is received. \a error will be
     * empty on valid reply.
     *
     * @param reply The response data. Semantics depends on the intent.
     * @param error Set to a valid error if an error occurred.
     */
    void replyReceived(const QJsonObject &reply, const Error &error);

private:
    Q_DECLARE_PRIVATE(IntentsInvoker)
    QScopedPointer<IntentsInvokerPrivate> d_ptr;
};

} // namespace RuntimeManager

#endif // RM_LIB_INTENTS_INVOKER_H
