// Copyright 2017 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef THIRD_PARTY_BLINK_RENDERER_PLATFORM_LOADER_FETCH_URL_LOADER_DEDICATED_OR_SHARED_WORKER_FETCH_CONTEXT_IMPL_H_
#define THIRD_PARTY_BLINK_RENDERER_PLATFORM_LOADER_FETCH_URL_LOADER_DEDICATED_OR_SHARED_WORKER_FETCH_CONTEXT_IMPL_H_

#include "base/strings/string_piece.h"
#include "base/synchronization/waitable_event.h"
#include "mojo/public/cpp/bindings/pending_receiver.h"
#include "mojo/public/cpp/bindings/receiver.h"
#include "mojo/public/cpp/bindings/remote.h"
#include "mojo/public/cpp/bindings/remote_set.h"
#include "mojo/public/cpp/bindings/shared_remote.h"
#include "services/network/public/mojom/url_loader_factory.mojom-forward.h"
#include "third_party/blink/public/mojom/loader/resource_load_info_notifier.mojom.h"
#include "third_party/blink/public/mojom/renderer_preference_watcher.mojom-blink.h"
#include "third_party/blink/public/mojom/service_worker/service_worker_container.mojom-shared.h"
#include "third_party/blink/public/mojom/service_worker/service_worker_worker_client.mojom-blink.h"
#include "third_party/blink/public/mojom/service_worker/service_worker_worker_client_registry.mojom-blink.h"
#include "third_party/blink/public/mojom/timing/worker_timing_container.mojom.h"
#include "third_party/blink/public/mojom/worker/subresource_loader_updater.mojom-blink.h"
#include "third_party/blink/public/platform/web_common.h"
#include "third_party/blink/public/platform/web_dedicated_or_shared_worker_fetch_context.h"
#include "third_party/blink/public/platform/web_string.h"
#include "third_party/blink/renderer/platform/wtf/casting.h"

namespace blink {

class ResourceLoadInfoNotifierWrapper;
class URLLoaderThrottleProvider;
class WeakWrapperResourceLoadInfoNotifier;
class WebFrameRequestBlocker;
class WebServiceWorkerProviderContext;
class WebSocketHandshakeThrottleProvider;

// This class is used for fetching resource requests from workers (dedicated
// worker and shared worker). This class is created on the main thread and
// passed to the worker thread. This class is not used for service workers. For
// service workers, ServiceWorkerFetchContextImpl class is used instead.
class BLINK_PLATFORM_EXPORT DedicatedOrSharedWorkerFetchContextImpl final
    : public WebDedicatedOrSharedWorkerFetchContext,
      public mojom::blink::SubresourceLoaderUpdater,
      public mojom::blink::ServiceWorkerWorkerClient,
      public mojom::blink::RendererPreferenceWatcher {
 public:
  // - |service_worker_client_receiver| receives OnControllerChanged()
  //   notifications.
  // - |service_worker_worker_client_registry| is used to register new
  //   ServiceWorkerWorkerClients, which is needed when creating a
  //   nested worker.
  //
  // Regarding the rest of params, see the comments on Create().
  DedicatedOrSharedWorkerFetchContextImpl(
      const RendererPreferences& renderer_preferences,
      mojo::PendingReceiver<mojom::blink::RendererPreferenceWatcher>
          preference_watcher_receiver,
      mojo::PendingReceiver<mojom::blink::ServiceWorkerWorkerClient>
          service_worker_client_receiver,
      mojo::PendingRemote<mojom::blink::ServiceWorkerWorkerClientRegistry>
          pending_service_worker_worker_client_registry,
      CrossVariantMojoRemote<mojom::ServiceWorkerContainerHostInterfaceBase>
          service_worker_container_host,
      std::unique_ptr<network::PendingSharedURLLoaderFactory>
          pending_loader_factory,
      std::unique_ptr<network::PendingSharedURLLoaderFactory>
          pending_fallback_factory,
      mojo::PendingReceiver<mojom::blink::SubresourceLoaderUpdater>
          pending_subresource_loader_updater,
      std::unique_ptr<URLLoaderThrottleProvider> throttle_provider,
      std::unique_ptr<WebSocketHandshakeThrottleProvider>
          websocket_handshake_throttle_provider,
      const WebVector<WebString>& cors_exempt_header_list,
      mojo::PendingRemote<mojom::ResourceLoadInfoNotifier>
          pending_resource_load_info_notifier);

  // WebDedicatedOrSharedWorkerFetchContext implementation:
  // Clones this fetch context for a nested worker.
  // For non-PlzDedicatedWorker. This will be removed once PlzDedicatedWorker is
  // enabled by default.
  scoped_refptr<WebDedicatedOrSharedWorkerFetchContext>
  CloneForNestedWorkerDeprecated(
      scoped_refptr<base::SingleThreadTaskRunner> task_runner) override;
  // For PlzDedicatedWorker. The cloned fetch context does not inherit some
  // fields (e.g., blink::WebServiceWorkerProviderContext) from this fetch
  // context, and instead that takes values passed from the browser process.
  scoped_refptr<WebDedicatedOrSharedWorkerFetchContext> CloneForNestedWorker(
      WebServiceWorkerProviderContext* service_worker_provider_context,
      std::unique_ptr<network::PendingSharedURLLoaderFactory>
          pending_loader_factory,
      std::unique_ptr<network::PendingSharedURLLoaderFactory>
          pending_fallback_factory,
      CrossVariantMojoReceiver<mojom::SubresourceLoaderUpdaterInterfaceBase>
          pending_subresource_loader_updater,
      scoped_refptr<base::SingleThreadTaskRunner> task_runner) override;
  // Sets properties associated with frames.
  // - For dedicated workers, the property is copied from the ancestor frame
  //   (directly for non-nested workers, or indirectly via its parent worker for
  //   nested workers).
  // - For shared workers, there is no parent frame, so the default value, or a
  //   value calculated in some way is set.
  //
  // TODO(nhiroki): Add more comments about security/privacy implications to
  // each property, for example, site_for_cookies and top_frame_origin.
  void set_ancestor_frame_id(int id) override;
  void set_frame_request_blocker(
      scoped_refptr<WebFrameRequestBlocker> frame_request_blocker) override;
  void set_site_for_cookies(
      const net::SiteForCookies& site_for_cookies) override;
  void set_top_frame_origin(const WebSecurityOrigin& top_frame_origin) override;

  // WebWorkerFetchContext implementation:
  void SetTerminateSyncLoadEvent(base::WaitableEvent*) override;
  void InitializeOnWorkerThread(AcceptLanguagesWatcher*) override;
  WebURLLoaderFactory* GetURLLoaderFactory() override;
  std::unique_ptr<WebURLLoaderFactory> WrapURLLoaderFactory(
      CrossVariantMojoRemote<network::mojom::URLLoaderFactoryInterfaceBase>
          url_loader_factory) override;
  std::unique_ptr<WebCodeCacheLoader> CreateCodeCacheLoader(
      CodeCacheHost*) override;
  void WillSendRequest(WebURLRequest&) override;
  mojom::ControllerServiceWorkerMode GetControllerServiceWorkerMode()
      const override;
  void SetIsOnSubframe(bool) override;
  bool IsOnSubframe() const override;
  net::SiteForCookies SiteForCookies() const override;
  absl::optional<WebSecurityOrigin> TopFrameOrigin() const override;
  void SetSubresourceFilterBuilder(
      std::unique_ptr<WebDocumentSubresourceFilter::Builder>) override;
  std::unique_ptr<WebDocumentSubresourceFilter> TakeSubresourceFilter()
      override;
  std::unique_ptr<WebSocketHandshakeThrottle> CreateWebSocketHandshakeThrottle(
      scoped_refptr<base::SingleThreadTaskRunner> task_runner) override;
  CrossVariantMojoReceiver<mojom::WorkerTimingContainerInterfaceBase>
  TakePendingWorkerTimingReceiver(int request_id) override;
  void SetIsOfflineMode(bool is_offline_mode) override;
  bool IsDedicatedWorkerOrSharedWorkerFetchContext() const override {
    return true;
  }

  // mojom::blink::ServiceWorkerWorkerClient implementation:
  void OnControllerChanged(mojom::ControllerServiceWorkerMode) override;

  // Sets the controller service worker mode.
  // - For dedicated workers (non-PlzDedicatedWorker), they depend on the
  //   controller of the ancestor frame (directly for non-nested workers, or
  //   indirectly via its parent worker for nested workers), and inherit its
  //   controller mode.
  // - For dedicated workers (PlzDedicatedWorker) and shared workers, the
  //   controller mode is passed from the browser processw when starting the
  //   worker.
  void set_controller_service_worker_mode(
      mojom::ControllerServiceWorkerMode mode);

  void set_client_id(const WebString& client_id);

  WebString GetAcceptLanguages() const override;

  // Sets up |receiver| to receive resource performance timings for the given
  // |request_id|. This receiver will be taken later by
  // TakePendingWorkerTimingReceiver().
  void AddPendingWorkerTimingReceiver(
      int request_id,
      mojo::PendingReceiver<mojom::WorkerTimingContainer> receiver);

  std::unique_ptr<ResourceLoadInfoNotifierWrapper>
  CreateResourceLoadInfoNotifierWrapper() override;

 private:
  class Factory;
  using WorkerTimingContainerReceiverMap =
      std::map<int /* request_id */,
               mojo::PendingReceiver<mojom::WorkerTimingContainer>>;

  ~DedicatedOrSharedWorkerFetchContextImpl() override;

  scoped_refptr<DedicatedOrSharedWorkerFetchContextImpl>
  CloneForNestedWorkerInternal(
      mojo::PendingReceiver<mojom::blink::ServiceWorkerWorkerClient>
          service_worker_client_receiver,
      mojo::PendingRemote<mojom::blink::ServiceWorkerWorkerClientRegistry>
          service_worker_worker_client_registry,
      CrossVariantMojoRemote<mojom::ServiceWorkerContainerHostInterfaceBase>
          service_worker_container_host,
      std::unique_ptr<network::PendingSharedURLLoaderFactory>
          pending_loader_factory,
      std::unique_ptr<network::PendingSharedURLLoaderFactory>
          pending_fallback_factory,
      mojo::PendingReceiver<mojom::blink::SubresourceLoaderUpdater>
          pending_subresource_loader_updater,
      scoped_refptr<base::SingleThreadTaskRunner> task_runner);

  // Resets the service worker url loader factory of a URLLoaderFactoryImpl
  // which was passed to Blink. The url loader factory is connected to the
  // controller service worker. Sets nullptr if the worker context is not
  // controlled by a service worker.
  void ResetServiceWorkerURLLoaderFactory();

  // Implements mojom::blink::SubresourceLoaderUpdater.
  void UpdateSubresourceLoaderFactories(
      std::unique_ptr<PendingURLLoaderFactoryBundle>
          subresource_loader_factories) override;

  // Implements mojom::blink::RendererPreferenceWatcher.
  void NotifyUpdate(const RendererPreferences& new_prefs) override;

  void ResetWeakWrapperResourceLoadInfoNotifier();

  // |receiver_| and |service_worker_worker_client_registry_| may be null if
  // this context can't use service workers. See comments for Create().
  mojo::Receiver<mojom::blink::ServiceWorkerWorkerClient> receiver_{this};
  mojo::Remote<mojom::blink::ServiceWorkerWorkerClientRegistry>
      service_worker_worker_client_registry_;

  // Bound to |this| on the worker thread.
  mojo::PendingReceiver<mojom::blink::ServiceWorkerWorkerClient>
      service_worker_client_receiver_;

  // Consumed on the worker thread to create
  // |service_worker_worker_client_registry_|.
  mojo::PendingRemote<mojom::blink::ServiceWorkerWorkerClientRegistry>
      pending_service_worker_worker_client_registry_;

  // Consumed on the worker thread to create |loader_factory_|.
  std::unique_ptr<network::PendingSharedURLLoaderFactory>
      pending_loader_factory_;
  // Consumed on the worker thread to create |fallback_factory_|.
  std::unique_ptr<network::PendingSharedURLLoaderFactory>
      pending_fallback_factory_;

  // This can be null if the |provider_context| passed to Create() was null or
  // already being destructed (see
  // content::ServiceWorkerProviderContext::OnNetworkProviderDestroyed()).
  CrossVariantMojoRemote<mojom::ServiceWorkerContainerHostInterfaceBase>
      service_worker_container_host_;

  mojom::ControllerServiceWorkerMode controller_service_worker_mode_ =
      mojom::ControllerServiceWorkerMode::kNoController;

  // The Client#id value of the shared worker or dedicated worker (since
  // dedicated workers are not yet service worker clients, it is the parent
  // document's id in that case). Passed to ControllerServiceWorkerConnector.
  WebString client_id_;

  // Initialized on the worker thread when InitializeOnWorkerThread() is called.
  // |loader_factory_| is used for regular loading by the worker. In
  // If the worker is controlled by a service worker, it creates a
  // ServiceWorkerSubresourceLoaderFactory instead.
  scoped_refptr<network::SharedURLLoaderFactory> loader_factory_;

  // Initialized on the worker thread when InitializeOnWorkerThread() is called.
  // If the worker is controlled by a service worker, it passes this factory to
  // ServiceWorkerSubresourceLoaderFactory to use for network fallback.
  scoped_refptr<network::SharedURLLoaderFactory> fallback_factory_;

  // Initialized on the worker thread when InitializeOnWorkerThread() is called.
  // Used to reconnect to the Network Service after the Network Service crash.
  // This is only used for dedicated workers when PlzDedicatedWorker is enabled.
  // When PlzDedicatedWorker is disabled, the ancestor render frame updates the
  // loaders via Host/TrackedChildURLLoaderFactoryBundle. For shared workers,
  // the renderer process detects the crash, and terminates the worker instead
  // of recovery.
  mojo::PendingReceiver<mojom::blink::SubresourceLoaderUpdater>
      pending_subresource_loader_updater_;
  mojo::Receiver<mojom::blink::SubresourceLoaderUpdater>
      subresource_loader_updater_{this};

  std::unique_ptr<WebDocumentSubresourceFilter::Builder>
      subresource_filter_builder_;
  // For dedicated workers, this is the ancestor frame (the parent frame for
  // non-nested workers, the closest ancestor for nested workers). For shared
  // workers, this is the shadow page.
  bool is_on_sub_frame_ = false;
  int ancestor_frame_id_ = MSG_ROUTING_NONE;
  // Set to non-null if the ancestor frame has an associated RequestBlocker,
  // which blocks requests from this worker too when the ancestor frame is
  // blocked.
  scoped_refptr<WebFrameRequestBlocker> frame_request_blocker_;
  net::SiteForCookies site_for_cookies_;
  absl::optional<url::Origin> top_frame_origin_;

  RendererPreferences renderer_preferences_;

  // |preference_watcher_receiver_| and |child_preference_watchers_| are for
  // keeping track of updates in the renderer preferences.
  mojo::Receiver<mojom::blink::RendererPreferenceWatcher>
      preference_watcher_receiver_{this};
  // Kept while staring up the worker thread. Valid until
  // InitializeOnWorkerThread().
  mojo::PendingReceiver<mojom::blink::RendererPreferenceWatcher>
      preference_watcher_pending_receiver_;
  mojo::RemoteSet<mojom::blink::RendererPreferenceWatcher>
      child_preference_watchers_;

  // This is owned by ThreadedMessagingProxyBase on the main thread.
  base::WaitableEvent* terminate_sync_load_event_ = nullptr;

  // The WebURLLoaderFactory which was created and passed to
  // Blink by GetURLLoaderFactory().
  std::unique_ptr<Factory> web_loader_factory_;

  std::unique_ptr<URLLoaderThrottleProvider> throttle_provider_;
  std::unique_ptr<WebSocketHandshakeThrottleProvider>
      websocket_handshake_throttle_provider_;

  WebVector<WebString> cors_exempt_header_list_;

  mojo::PendingRemote<mojom::ResourceLoadInfoNotifier>
      pending_resource_load_info_notifier_;

  // Used to notify the loading stats by ResourceLoadInfo struct for dedicated
  // workers.
  mojo::Remote<mojom::ResourceLoadInfoNotifier> resource_load_info_notifier_;

  // Wrap a raw blink::mojom::ResourceLoadInfoNotifier pointer directed at
  // |resource_load_info_notifier_|'s receiver.
  std::unique_ptr<WeakWrapperResourceLoadInfoNotifier>
      weak_wrapper_resource_load_info_notifier_;

  AcceptLanguagesWatcher* accept_languages_watcher_ = nullptr;

  // Contains pending receivers whose corresponding requests are still
  // in-flight. The pending receivers are taken by
  // TakePendingWorkerTimingReceiver() when the request is completed.
  WorkerTimingContainerReceiverMap worker_timing_container_receivers_;

  base::WeakPtrFactory<DedicatedOrSharedWorkerFetchContextImpl> weak_factory_{
      this};
};

template <>
struct DowncastTraits<DedicatedOrSharedWorkerFetchContextImpl> {
  static bool AllowFrom(const WebWorkerFetchContext& context) {
    return context.IsDedicatedWorkerOrSharedWorkerFetchContext();
  }
};

}  // namespace blink

#endif  // THIRD_PARTY_BLINK_RENDERER_PLATFORM_LOADER_FETCH_URL_LOADER_DEDICATED_OR_SHARED_WORKER_FETCH_CONTEXT_IMPL_H_
