
// SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: Proprietary

/* 
 * This file is part of the Aurora OS SDK
 */

// Keep these in sync with org.merproject.emulator [[Begin
const emulatorVmName = "latest" === "ea"
    ? "AuroraOS-5.2.0.75EA"
    : "AuroraOS-5.2.0.75";
const emulatorVmSnapshotName = "5.2.0.75";
const emulatorMemoryMB = "4096";
const emulatorCpuCount = "2";
// End]]

const emulatorVmUri = "sfdkvm:Qemu#" + emulatorVmName;
const emulatorProductName = "Aurora OS";
const emulatorProductRelease = "latest" === "ea"
    ? "5.2.0.75EA"
    : "5.2.0.75";

const emulatorIdx = 0;
const sshPort = 2223 + emulatorIdx;
const freePortsRangeStart = 20000 + 20 * emulatorIdx;
const freePortsRangeEnd = 20019 + 20 * emulatorIdx;
const qmlLivePort = 10234 + emulatorIdx;

if (emulatorIdx > 14)
    throw "Too many emulators";

const sharedSshDir =  "@TargetDir@/emulator/" + emulatorVmName + "/ssh";
const sharedConfigDir = "@TargetDir@/emulator/" + emulatorVmName + "/vmshare";
const sharedMediaDir = "@TargetDir@/emulator/media";

const deviceModelDconfDbFile = sharedConfigDir + "/device-model.txt";
const deviceModelEglfsConfigFile = sharedConfigDir + "/65-emul-wayland-ui-scale.conf";
const compositorConfigFile = sharedConfigDir + "/60-emul-wayland-ui.conf";
const qemuTouchRulesFile = sharedConfigDir + "/99-qemu-touch.rules";

// we don't have to care about version numbers smaller than 3.0
const userName = "defaultuser";


function Component()
{
    console.log("Component constructor: " + component.name);
    component.setValue("DisplayName", qsTranslate("installscript", "%1-%2%3 Emulator Integration for Aurora IDE")
        .arg("AuroraOS")
        .arg("latest")
        .arg("latest" != "5.2.0.75" ? " (5.2.0.75)" : ""));
}

Component.prototype.createOperations = function()
{
    // Call the base createOperations
    component.createOperations();

    if (!installer.isUninstaller()) {
        installComponent()
    }
    else {
        removeMerEmulatorFromQtCreator(emulatorVmUri);
    }
}

function installComponent()
{
    const settings = function(key, value) { 
        const emulatorSettings = "@TargetDir@/emulator/" + emulatorVmName + "/emulator-setup.ini";
        component.addOperation("Settings", "path=" + emulatorSettings,
                               "method=set", "key=" + key, "value=" + value);
    }

    const setPortForwarding = function(ruleName, hostPort, guestPort, protocol) {
        settings("ports_forwarding/" + ruleName + "/host", hostPort);
        settings("ports_forwarding/" + ruleName + "/guest", guestPort);
        settings("ports_forwarding/" + ruleName + "/protocol", protocol);
    }
    
    setPortForwarding("ssh", sshPort, 22, "tcp");
    setPortForwarding("qmllive_1", qmlLivePort, qmlLivePort, "tcp");

    for (var freePort = freePortsRangeStart; freePort <= freePortsRangeEnd; freePort++) {
        setPortForwarding("freeport_" + freePort, freePort, freePort, "tcp");
    }

    const setSharedDir = function(tag, sharedDir) {
        settings("folders/" + tag, sharedDir);
    }

    setSharedDir("ssh", sharedSshDir);
    setSharedDir("configs", sharedConfigDir);
    setSharedDir("media", sharedMediaDir);

    settings("system/cpus", emulatorCpuCount);
    settings("system/mac", "52:54:00:12:34:56");
    settings("system/memory", emulatorMemoryMB);

    const qemuImage = systemInfo.kernelType === "winnt" ? "qemu-img.exe" : "qemu-img";
    const emulatorTargetDir = "@TargetDir@/emulator/" + emulatorVmName;
    component.addOperation("Execute", ["{0}", installer.value("qemuPath") + "/" + qemuImage,
                                       "snapshot", "-c", emulatorVmSnapshotName, emulatorTargetDir + "/image.qcow2"]);

    component.addOperation("Mkdir", sharedConfigDir);
    component.addOperation("Mkdir", sharedMediaDir);

    deploySshKeys();

    var initialDeviceModel = installer.componentByName("org.merproject.emulators").initialDeviceModel;

    settings("display/width", initialDeviceModel.hres_px / initialDeviceModel.scaleDownFactor());
    settings("display/height", initialDeviceModel.vres_px / initialDeviceModel.scaleDownFactor());

    component.addOperation("Delete", deviceModelDconfDbFile);
    component.addOperation("AppendFile", deviceModelDconfDbFile, initialDeviceModel.dconfDb);
    component.addOperation("Delete", deviceModelEglfsConfigFile);
    component.addOperation("AppendFile", deviceModelEglfsConfigFile, initialDeviceModel.eglfConfig());
    component.addOperation("Delete", compositorConfigFile);
    component.addOperation("AppendFile", compositorConfigFile, initialDeviceModel.compositorConfig());
    component.addOperation("Delete", qemuTouchRulesFile);
    component.addOperation("AppendFile", qemuTouchRulesFile, initialDeviceModel.qemuTouchRules());

    addMerEmulatorToQtCreator(initialDeviceModel);
}

function removeMerEmulatorFromQtCreator(vmUri)
{
    const installDir = installer.value("TargetDir");
    const sdkTool = installer.value("SDKToolBinary");
    const errorMessage = "errormessage=" +  qsTranslate("installscript", "Could not remove the device \"%1\"").arg(emulatorVmName);
    component.addOperation("Execute", ["{0,2,3}", sdkTool, "rmSfdkEmulator",
                                       "--vm-uri", vmUri, errorMessage]);
}

function addMerEmulatorToQtCreator(initialDeviceModel)
{
    const installDir = installer.value("TargetDir");
    const sdkTool = installer.value("SDKToolBinary");
    removeMerEmulatorFromQtCreator(emulatorVmUri);
    component.addOperation("Execute", ["{0,1}", sdkTool, "addSfdkEmulator",
                                       "--installdir", installDir,
                                       "--product-name", emulatorProductName,
                                       "--product-release", emulatorProductRelease,
                                       "--vm-uri", emulatorVmUri,
                                       "--vm-factory-snapshot", emulatorVmSnapshotName,
                                       "--autodetected", "true",
                                       "--shared-ssh", sharedSshDir,
                                       "--shared-config", sharedConfigDir,
                                       "--host", "127.0.0.1",
                                       "--username", userName,
                                       "--private-key-file", "@merSdkSshKeyFile@",
                                       "--ssh-port", sshPort,
                                       "--qml-live-ports", qmlLivePort,
                                       "--free-ports", freePortsRangeStart + "-" + freePortsRangeEnd,
                                       "--mac", "52:54:00:12:34:56", // The default MAC for a qemu vm
                                       "--subnet", "10.220.220" , // Do nothing for the qemu emulator
                                       "--device-model", initialDeviceModel.name,
                                       "--view-scaled", initialDeviceModel.isScaledDown() ? "true" : "false",
                                       "UNDOEXECUTE",
                                       "{0,1}", sdkTool, "rmSfdkEmulator",
                                       "--vm-uri", emulatorVmUri
                           ]);
}

function deploySshKeys()
{
    var sharedSshDirUser = sharedSshDir + "/" + userName + "/";
    var sharedSshDirRoot = sharedSshDir + "/root/";

    component.addOperation("Mkdir", sharedSshDirUser);
    component.addOperation("Mkdir", sharedSshDirRoot);

    //todo: restore
    var authorizedKeysBaseName = "authorized_keys";
    component.addOperation("Copy", "@merSdkSshKeyPublicFile@", sharedSshDirUser + authorizedKeysBaseName);
    component.addOperation("Copy", "@merSdkSshKeyPublicFile@", sharedSshDirRoot + authorizedKeysBaseName);
}
