/****************************************************************************
**
** Copyright (c) 2021 Open Mobile Platform LLC
**
****************************************************************************/
import QtQuick 2.0
import Sailfish.Silica 1.0
import ru.omp.qca.private 1.0

Item {
    id: root

    property alias iconSource: iconState.source
    property bool visibleSuccessful: true
    property bool visibleBackButton: true
    property Page realParent: null

    property int buttonWidth: Screen.sizeCategory <= Screen.Medium
                              ? parent.width - Theme.horizontalPageMargin*2
                              : Theme.buttonWidthLarge

    signal back
    signal done

    anchors.fill: parent
    visible: false

    function wait() {      
        state = WaitState.waiting
        timerTimeout.start()
        visible = true
        timerClose.stop()
    }

    function successful(message) {
        if (state === WaitState.waiting) {
            timerTimeout.stop()
            timerClose.start()
            visible = true

            if (visibleSuccessful) {
                state = WaitState.sucessfull
                labelMessage.text = message
            } else {
                state = WaitState.waiting
            }
        }
    }

    function warning(message) {
        state = WaitState.warning
        labelMessage.text = message
        timerTimeout.stop()
        visible = true
    }

    MouseArea {
        enabled: root.visible
        anchors.fill: root
    }

    Item {
        id: imageState

        anchors.centerIn: parent
        width: Math.max(iconState.width, busyIndicator.width)
        height: Math.max(iconState.height, busyIndicator.height)

        Icon {
            id: iconState

            anchors.centerIn: parent
            source: ""
        }

        BusyIndicator {
            id: busyIndicator

            anchors.centerIn: parent
            running: visible
            size: BusyIndicatorSize.Large
            visible: root.state === WaitState.waiting
        }
    }

    Label {
        id: labelMessage

        anchors {
            top: imageState.bottom
            topMargin: Theme.paddingLarge
        }

        x: Theme.horizontalPageMargin
        width: parent.width - x*2
        wrapMode: Text.Wrap
        horizontalAlignment: Text.AlignHCenter
        maximumLineCount: 5
    }

    Button {
        anchors {
            bottom: parent.bottom
            bottomMargin: Theme.paddingLarge
            horizontalCenter: parent.horizontalCenter
        }
        visible: visibleBackButton && root.state === WaitState.warning
        //% "Back"
        text: qsTrId("qca-bt-back")
        preferredWidth: buttonWidth
        onClicked: back()
    }

    states: [
        State {
            name: WaitState.sucessfull
            PropertyChanges {
                target: root
                iconSource: "image://theme/icon-l-acknowledge"
            }
        },
        State {
            name: WaitState.waiting
            PropertyChanges {
                target: root
                iconSource: ""
            }
            PropertyChanges {
                target: labelMessage
                text: ""
            }
        },
        State {
            name: WaitState.warning
            PropertyChanges {
                target: root
                iconSource: "image://theme/icon-l-attention"
            }
        }
    ]

    Timer {
        id: timerTimeout
        interval: 30 * 1000

        onTriggered: {
            //% "Time out"
            warning(qsTrId("qca-la-timeout_warning"))
        }
    }

    Timer {
        id: timerClose
        interval: 2500

        onTriggered: {
            done()
            visible = false
        }
    }

    BlockingOverlay {
        enabled: root.visible
        targetPage: _findPage()
        anchors.fill: parent
    }

    function _findPage() {
        var r = realParent
        while (r && !r.hasOwnProperty('__silica_page'))
            r = r.parent
        return r
    }

    Component.onCompleted: {
        realParent = parent
        parent = realParent.parent
    }
}
