/****************************************************************************
**
** SPDX-FileCopyrightText: 2021-2024 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: Proprietary
**
****************************************************************************/
import QtQuick 2.0
import Sailfish.Silica 1.0
import ru.omp.qca 1.0
import ru.omp.qca.private 1.0

Page {
    id: page

    property string fileName
    property string initialPath
    property string description
    property int type
    property int error
    property bool buttonClicked: false
    property bool errorStatus
    property bool isExport

    function backToCertificateView(operationType) {
        var certificatePage = pageStack.find(function(page) { return page.hasOwnProperty('__hasPersonal') })
        pageStack.pop(certificatePage, operationType)
    }

    onStatusChanged: {
        if (status == PageStatus.Active && buttonClicked) {
            backToCertificateView(PageStackAction.Immediate)
        }
    }

    backNavigation: false

    Column {
        id: column

        anchors.verticalCenter: parent.verticalCenter
        width: parent.width
        spacing: Theme.paddingLarge

        Icon {
            anchors.horizontalCenter: parent.horizontalCenter
            color: Theme.secondaryHighlightColor
            source: !errorStatus ? "image://theme/icon-l-acknowledge" : "image://theme/icon-l-attention"
        }

        Label {
            anchors.horizontalCenter: parent.horizontalCenter
            width: parent.width - Theme.horizontalPageMargin * 2
            font.pixelSize: Theme.fontSizeMediumBase
            horizontalAlignment: Text.AlignHCenter
            wrapMode: Text.Wrap
            color: Theme.highlightColor
            text: {
                if (isExport) {
                    //% "%1 %2 saved" : "Unable to save %1 %2 to chosen location"
                    var exportDescription = !errorStatus ? qsTrId("qca-la-status_export_save") : qsTrId("qca-la-status_export_fail")
                    switch (type) {
                    case KeyStoreModel.Request:
                        //% "Certificate request"
                        exportDescription.arg(qsTrId("qca-la-status_request")).arg(fileName)
                        break
                    // Otherwise "Certificate" should be shown
                    default:
                        //% "Certificate"
                        exportDescription.arg(qsTrId("qca-la-status_imported")).arg(fileName)
                        break
                    }
                } else {
                    switch (type) {
                    case KeyStoreModel.Request:
                        //% "Request %1 was created"
                        !errorStatus ? qsTrId("qca-la-status_request_created").arg(fileName) : ErrorHints.certErrorToString(type, fileName, error)
                        break
                    case KeyStoreModel.Imported:
                        //% "Certificate %1 was added to your device"
                        !errorStatus ? qsTrId("qca-la-status_cert_imported").arg(fileName) : ErrorHints.certErrorToString(type, fileName, error)
                        break
                    case KeyStoreModel.RootImported:
                        //% "Root certificate %1 was added to your device"
                        !errorStatus ? qsTrId("qca-la-status_root_cert_imported").arg(fileName) : ErrorHints.certErrorToString(type, fileName, error)
                        break
                    case KeyStoreModel.SmartCard:
                        //% "Personal certificate %1 was added to your device"
                        !errorStatus ? qsTrId("qca-la-status_personal_cert_imported").arg(fileName) : ErrorHints.certErrorToString(type, fileName, error)
                        break
                    }
                }
            }
        }

        Label {
            anchors.horizontalCenter: parent.horizontalCenter
            width: parent.width - Theme.horizontalPageMargin * 2
            font.pixelSize: Theme.fontSizeMediumBase
            horizontalAlignment: Text.AlignHCenter
            wrapMode: Text.Wrap
            color: Theme.secondaryHighlightColor
            //% "For further work with the certificate, you need to send it to the Certification Center for processing."
            text: !errorStatus ? qsTrId("qca-la-request_info") : page.description
            visible: type === KeyStoreModel.Request && !isExport
        }

        Button {
            anchors.horizontalCenter: parent.horizontalCenter
            width: Theme.buttonWidthLarge
            height: Theme.itemSizeMedium
            //% "Show in folder"
            text: qsTrId("qca-bu-show_folder")
            visible: (type === KeyStoreModel.Request || isExport) && !errorStatus
            onClicked: {
                pageStack.animatorPush("Sailfish.FileManager.DirectoryPage",
                                       { 'initialPath': initialPath,
                                           //% "Certificates"
                                           'title': qsTrId("qca-la-certificates") })
                timerClose.stop()
                buttonClicked = true
            }
        }

        Timer {
            id: timerClose

            interval: 3 * 1000
            running: true
            onTriggered: {
                backToCertificateView(PageStackAction.Animated)
            }
        }
    }
}
