/****************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2021-2025 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: Proprietary
**
****************************************************************************/
import QtQuick 2.0
import Sailfish.Silica 1.0
import ru.omp.qca 1.0
import ru.omp.authclient 1.0

Dialog {
    id: root

    property int type
    property string fileName
    property string nameAndSurname
    property string email
    property string entryId
    property Page _removeBusyPage
    property bool isRoot

    function backToCertificateView() {
        var certificatePage = pageStack.find(function(page) { return page.hasOwnProperty('__hasPersonal') })
        pageStack.pop(certificatePage)
    }

    onAccepted: {
        query.authenticate(function() {
            if (type !== KeyStoreModel.Request && type !== KeyStoreModel.System) {
                _removeBusyPage.wait()
                KeyStoreModel.removeCertificate(root.entryId)
            } else if (type === KeyStoreModel.Request) {
                _removeBusyPage.stop()
            }
        }, function() {
            backToCertificateView()
        })
    }

    acceptDestination: _removeBusyPage
    onRejected: pageStack.pop()
    onStatusChanged: {
        if (status === PageStatus.Active) {
            if (_removeBusyPage != null) {
                _removeBusyPage.destroy()
                _removeBusyPage = null
            }

            _removeBusyPage = removeBusyComponent.createObject(root)
        }
    }

    Component {
        id: removeBusyComponent
        BusyPage {
            id: busyPage

            //% "Removing a certificate"
            labelText: qsTrId("qca-la-cert_removing")
            onDone: backToCertificateView()
        }
    }

    Connections {
        target: KeyStoreModel
        onEntryRemovedChanged: {
            if (success)
                _removeBusyPage.successful()

            backToCertificateView()
        }
    }

    DialogHeader {
        id: dialogHeader

        title: {
            //% "Do you want remove %1 %2 certificate from the device?"
            var crtDescription = qsTrId("qca-la-remove_crt_description")

            switch (type) {
            case KeyStoreModel.Request:
                //% "Delete %1 request for creating a personal certificate?"
                qsTrId("qca-la-remove_request").arg(fileName)
                break
            case KeyStoreModel.SmartCard:
                //% "personal"
                crtDescription.arg(qsTrId("qca-la-remove_personal")).arg(fileName)
                break
            case KeyStoreModel.RootImported:
                //% "root"
                crtDescription.arg(qsTrId("qca-la-remove_root")).arg(fileName)
                break
            case KeyStoreModel.Imported:
                //% "imported"
                crtDescription.arg(qsTrId("qca-la-remove_imported")).arg(fileName)
                break
            case KeyStoreModel.User:
                if (isRoot)
                    //% "root"
                    crtDescription.arg(qsTrId("qca-la-remove_root")).arg(fileName)
                else
                    //% "user"
                    crtDescription.arg(qsTrId("qca-la-remove_user")).arg(fileName)
                break
            }
        }
    }

    Label {
        anchors.top: dialogHeader.bottom
        width: parent.width
        leftPadding: Theme.horizontalPageMargin
        rightPadding: Theme.horizontalPageMargin
        color: Theme.primaryColor
        wrapMode: Text.Wrap
        //% "Removing root certificate will compromise entire chain of trust"
        text: qsTrId("qca-la-remove_root_crt_warning")
        visible: isRoot
    }

    AuthQuery {
        id: query
    }
}
