/****************************************************************************
**
** Copyright (c) 2021 - 2023. Open Mobile Platform LLC
**
****************************************************************************/
pragma Singleton
import QtQuick 2.2
import ru.omp.qca 1.0

QtObject {
    // SecureMessage Error
    function smErrorToString(errorCode) {
        switch (errorCode) {
        case SecureMessage.NoError:
            //% "No error"
            return qsTrId("qca-me-error_no")
        case SecureMessage.ErrorPathNotFound:
            //% "File not found"
            return qsTrId("qca-me-error_file_not_found")
        case SecureMessage.ErrorFileNotLoaded:
            //% "File not loaded"
            return qsTrId("qca-me-error_file_not_loaded")
        case SecureMessage.ErrorPassphrase:
            //% "Invalid passphrase."
            return qsTrId("qca-me-error_invalid_passphrase")
        case SecureMessage.ErrorFormat:
            //% "Bad input format."
            return qsTrId("qca-me-error_bad_input_format")
        case SecureMessage.ErrorSignerExpired:
            //% "Signer key is expired."
            return qsTrId("qca-me-error_signer_key_is_expired")
        case SecureMessage.ErrorSignerInvalid:
            //% "Signer key is invalid."
            return qsTrId("qca-me-error_invalid_signer_key_is_invalid")
        case SecureMessage.ErrorEncryptExpired:
            //% "Encrypting key is expired."
            return qsTrId("qca-me-error_encrypting_key_is_expired")
        case SecureMessage.ErrorEncryptUntrusted:
            //% "Encrypting key is untrusted."
            return qsTrId("qca-me-error_encrypting_key_is_untrusted")
        case SecureMessage.ErrorEncryptInvalid:
            //% "Encrypting key is invalid."
            return qsTrId("qca-me-error_encrypting_key_is_invalid")
        case SecureMessage.ErrorNeedCard:
            //% "Card was needed but not found."
            return qsTrId("qca-me-error_card_was_need_but_not_found")
        case SecureMessage.ErrorCertKeyMismatch:
            //% "Certificate and private key don't match."
            return qsTrId("qca-me-error_certificate_and_private_key_do not match")
        case SecureMessage.ErrorUnknown:
        default:
            //% "General error."
            return qsTrId("qca-me-error_general")
        }
    }

    function validityToString(validity) {
        switch (validity) {
        case Certificate.ValidityGood:
            //% "Validated"
            return qsTrId("qca-me-validate")
        case Certificate.ErrorRejected:
            //% "Root CA is marked to reject the specified purpose"
            return qsTrId("qca-me-error_root_ca_is_marked")
        case Certificate.ErrorUntrusted:
            //% "Certificate not trusted for the required purpose"
            return qsTrId("qca-me-error_certificate_not_trusted")
        case Certificate.ErrorSignatureFailed:
            //% "Invalid signature"
            return qsTrId("qca-me-error_invalid_signature")
        case Certificate.ErrorInvalidCA:
            //% "Invalid CA certificate"
            return qsTrId("qca-me-error_invalid_ca_certificate")
        case Certificate.ErrorInvalidPurpose:
            //% "Invalid certificate purpose"
            return qsTrId("qca-me-error_invalid_certificate_purpose")
        case Certificate.ErrorSelfSigned:
            //% "Certificate is self-signed"
            return qsTrId("qca-me-error_certificate_is_self_signed")
        case Certificate.ErrorRevoked:
            //% "Certificate has been revoked"
            return qsTrId("qca-me-error_certificate_has_been_revoked")
        case Certificate.ErrorPathLengthExceeded:
            //% "Maximum certificate chain length exceeded"
            return qsTrId("qca-me-error_maximum_certificate_chain_length")
        case Certificate.ErrorExpired:
            //% "Certificate has expired"
            return qsTrId("qca-me-error_certificate_has_expired")
        case Certificate.ErrorExpiredCA:
            //% "CA has expired"
            return qsTrId("qca-me-error_ca_has_expired")
        case Certificate.ErrorValidityUnknown:
        default:
            //% "General certificate validation error"
            return qsTrId("qca-me-error_certificate_validation")
        }
    }

    function loadErrorToString(e) {
        switch (e) {
        case LoaderFile.ErrorNotOpenFile:
            //% "Not open file"
            return qsTrId("qca-me-error_not_open_file")
        case LoaderFile.ErrorNotLoadFile:
            //% "Not load file"
            return qsTrId("qca-me-error_not_load_file")
        case LoaderFile.ErrorMimeType:
            //% "The Mime type not support"
            return qsTrId("qca-me-error_the_mime_type_not_support")
        default:
            //% "Unknown error load"
            return qsTrId("qca-me-error_unknown_load")
        }
    }
    // Sign Verify Error
    function svErrorToString(e) {
        switch (e) {
        case SignVerify.ErrorFileNotOpened:
            //% "Not opened file"
            return qsTrId("qca-me-error_not_opened_file")
        case SignVerify.ErrorNotFoundCmsFile:
            //% "Don't found CMS file"
            return qsTrId("qca-me-error_not_found_cms_file")
        case SignVerify.ErrorSign:
            //% "Error sign"
            return qsTrId("qca-me-error_sign")
        case SignVerify.ErrorVerify:
            //% "Error verify"
            return qsTrId("qca-me-error_verify")
        default:
            //% "Unknown error SignVerify"
            return qsTrId("qca-me-error_unknown_sign_verify")
        }
    }
    // Certificate Error Message
    function certErrorMessage(error) {
        switch (error) {
            case KeyStoreModel.ErrorDecode:
                //% "Failed to decode"
                return qsTrId("qca-me-error_cert_error_decode")
            case KeyStoreModel.ErrorPassphrase:
                //% "Invalid passphrase"
                return qsTrId("qca-me-error_cert_error_passphrase")
            case KeyStoreModel.ErrorFile:
                //% "Failed to open file"
                return qsTrId("qca-me-error_cert_error_file")
            default:
                //% "Unknown error"
                return qsTrId("qca-me-error_cert_error_unknown")
        }
    }
    // Certificate/Request Error
    function certErrorToString(type, fileName, error) {
        var errorStr = certErrorMessage(error)
        switch (type) {
        case KeyStoreModel.Request:
            //% "Request %1 was not created: %2"
            return qsTrId("qca-me-error_request_not_created").arg(fileName).arg(errorStr)
        case KeyStoreModel.Imported:
            //% "Certificate %1 was not added to your device: %2"
            return qsTrId("qca-me-error_cert_not_imported").arg(fileName).arg(errorStr)
        case KeyStoreModel.RootImported:
            //% "Root certificate %1 was not added to your device: %2"
            return qsTrId("qca-me-error_root_cert_not_imported").arg(fileName).arg(errorStr)
        case KeyStoreModel.SmartCard:
            //% "Personal certificate %1 was not added to your device: %2"
            return qsTrId("qca-me-error_personal_cert_not_imported").arg(fileName).arg(errorStr)
        }
    }
}
