/****************************************************************************
**
** Copyright (c) 2021 Open Mobile Platform LLC
**
****************************************************************************/
import QtQuick 2.0
import Sailfish.Silica 1.0
import ru.omp.qca 1.0
import "../"

Dialog {
    id: root

    property Page _keyBusyPage

    acceptDestination: pinInputComponent
    canAccept: request.prKeyLocationCount > 0
               && request.prKeyAlgorithmCount > 0
               && request.keyIdentifier.length > 0
               && request.keyMarker.length > 0

    onStatusChanged: {
        if (status === PageStatus.Active) {
            if (_keyBusyPage != null) {
                _keyBusyPage.destroy()
                _keyBusyPage = null
            }

            _keyBusyPage = keyBusyComponent.createObject(root)
        }
    }

    DialogHeader {
        id: dialogHeader

        //% "Create new key"
        title: qsTrId("qca-la-create_key")
    }

    Component {
        id: pinInputComponent

        PinInput {
            acceptDestination: _keyBusyPage
            onAccepted: {
                if (pin.length > 0) {
                    request.prKeyPin = pin
                    request.createKeyPair()
                    _keyBusyPage.wait()
                }
            }
            onRejected: request.pinRejected()
        }
    }

    Component {
        id: keyBusyComponent

        BusyPage {
            id: busyPage

            //% "Key generation"
            labelText: qsTrId("qca-la-generation")
            // Increasing the interval for generating an rsa key pair
            waitingKeystoreTimerInterval: 180 * 1000
            onDone: pageStack.push(Qt.resolvedUrl("TypeAndSubjectDialog.qml"), { request: request })
        }
    }

    CertificateRequest {
        id: request

        property int prKeyLocationCount: 0
        property int prKeyAlgorithmCount: 0

        keyStoreModel: KeyStoreModel
        onCreateKeyPairDone: _keyBusyPage.successful()

        onRequestCreated: {
            pageStack.push("StatusPage.qml", {
                               initialPath: request.requestPath(),
                               fileName: request.fileName,
                               type: KeyStoreModel.Request
                           })
        }

        Component.onCompleted: {
            request.init()
            request.prKeyLocationCount = request.prKeyLocation.rowCount()
            request.prKeyAlgorithmCount = request.prKeyAlgorithm.rowCount()
        }
    }

    Column {
        anchors.top: dialogHeader.bottom
        width: parent.width

        TextField {
            width: parent.width
            //% "Key identifier"
            placeholderText: qsTrId("qca-ph-key_id")
            label: placeholderText
            onTextChanged: request.keyIdentifier = text
        }

        TextField {
            width: parent.width
            //% "Marker"
            placeholderText: qsTrId("qca-ph_marker")
            label: placeholderText
            onTextChanged: request.keyMarker = text
        }

        ComboBox {
            width: parent.width
            //% "Key algorithm"
            label: qsTrId("qca-la-key_algorithm")
            menu: ContextMenu {
                Repeater {
                    model: request.prKeyAlgorithm
                    MenuItem {
                        text: model.name
                        onClicked: {
                            request.prKeyAlgorithmId = model.id
                        }
                    }
                }
            }
        }

        ComboBox {
            width: parent.width
            //% "Closed key location"
            label: qsTrId("qca-la-closed_key_location")
            menu: ContextMenu {
                Repeater {
                    model: KeyStoreModel.certLocation
                    MenuItem {
                        textFormat: Text.StyledText
                        color: highlighted ? Theme.highlightColor : Theme.primaryColor
                        text: ("%1 %2").arg(model.name).arg(model.type)
                        onClicked: request.setPrKeyLocationIndex(model.index)
                    }
                    Component.onCompleted: request.setPrKeyLocationIndex(model.index)
                }
            }
        }
    }
}
