/****************************************************************************
**
** Copyright (c) 2020 - 2021 Open Mobile Platform LLC
**
****************************************************************************/
import QtQuick 2.3
import QtQuick.Layouts 1.1
import Sailfish.Silica 1.0
import ru.omp.qca 1.0
import "private"

Page {
    id: page

    property string path
    property Component viewPdf
    property var keyEntry
    property int mode: SignVerify.SignMode

    readonly property bool isPhone: Screen.sizeCategory == Screen.Medium
    readonly property var availableMimeTypes: loaderFile.availableMimeTypes

    property bool _inProcess

    SilicaFlickable {
        anchors.fill: parent
        contentHeight: column.height + Theme.paddingLarge
        visible: loaderFile.status === LoaderFile.Ready
        Column {
            id: column
            width: parent.width
            spacing: Theme.paddingLarge
            PageHeader {
                id: header
                titleColor: palette.primaryColor
                title: Theme.highlightText(fileInfo.name, loaderFile.fileName, palette.highlightColor)
                _titleItem.truncationMode: TruncationMode.Elide
            }

            FileInfo {
                id: fileInfo
                loaderFile: loaderFile
                viewPdf: page.viewPdf
            }

            ComboBox {
                id: keys
                property var keyEntry: page.keyEntry
                //% "Electronic signature"
                label: qsTrId("qca-la-electronic_signature")
                enabled: page.keyEntry === undefined
                menu: ContextMenu {
                    Repeater {
                        id: contextRepeater

                        model: certificateModel

                        MenuItem {
                            text: model.name

                            onClicked: keys.keyEntry = model.keyStoreEntry
                            Component.onCompleted: if (index === 0) keys.keyEntry = model.keyStoreEntry
                        }
                    }
                }
                BusyIndicator {
                    anchors {
                        verticalCenter: parent.verticalCenter
                        right: parent.right
                        rightMargin: Theme.paddingLarge
                    }
                    size: BusyIndicatorSize.ExtraSmall
                    running: contextRepeater.count === 0
                }
            }

            CertificateBaseInfo {
                certificate: keys.keyEntry.certificate
            }

            Button {
                id: signButton

                anchors.horizontalCenter: parent.horizontalCenter
                preferredWidth: isPortrait && isPhone ? parent.width - Theme.horizontalPageMargin * 2 : Theme.buttonWidthLarge

                text: mode === SignVerify.SignMode
                      //% "Sign"
                      ? qsTrId("qca-bt-sign")
                      //% "Verify"
                      : qsTrId("qca-bt-verify")
                enabled: keys.keyEntry !== undefined && !KeyStoreModel.isBusy
                onClicked: {
                    var dialog = pageStack.push(Qt.resolvedUrl("PinInput.qml"))
                    dialog.accepted.connect(function() {
                        waitLayer.wait()
                        signMessage.pinAccepted(dialog.pin)
                        _inProcess = false
                        if (mode === SignVerify.SignMode)
                            signMessage.sign(loaderFile.rawData, loaderFile.filePath + ".cms", keys.keyEntry)
                        else
                            signMessage.verify(loaderFile.rawData, loaderFile.filePath + ".cms", keys.keyEntry)
                    })

                    dialog.rejected.connect(function() {
                        waitLayer.wait()
                        signMessage.pinRejected()
                    })
                }
            }
        }
        VerticalScrollDecorator {}
    }

    WaitLayer {
        id: waitLayer

        visibleSuccessful: false
        onBack: waitLayer.visible = false
    }

    Loader {
        id: errorLoader
        anchors.fill: parent
        source: page.status === PageStatus.Active && loaderFile.error != LoaderFile.NoError ? "private/FullscreenError.qml" : ""
        onLoaded: item.error = ErrorHints.loadErrorToString(loaderFile.error)
    }

    FullscreenError {
        id: fullscreenError
    }

    LoaderFile {
        id: loaderFile
        filePath: page.path
    }

    SignVerify {
        id: signMessage

        onErrorChanged: waitLayer.warning(ErrorHints.svErrorToString(error))
        onSignSuccess: signMessage.verify(loaderFile.rawData, loaderFile.filePath + ".cms", keys.keyEntry)
        onVerifySuccess: {
            _inProcess = true
            pageStack.replace(Qt.resolvedUrl("InfoSignPage.qml"), {
                                  "fileName": loaderFile.fileName,
                                  "certificate": keys.keyEntry.certificate,
                                  "mode": mode
                              }, PageStackAction.Immediate)
        }
    }

    CertificateModel {
        id: certificateModel
        sourceModel: KeyStoreModel
        filterRole: KeyStoreModel.SmartCard
    }

    onStatusChanged: {
        if (status == PageStatus.Inactive && _inProcess)
            waitLayer.successful()
    }
}
