/****************************************************************************
**
** Copyright (c) 2021 Open Mobile Platform LLC
**
****************************************************************************/
import QtQuick 2.6
import Sailfish.Silica 1.0

Dialog {
    id: root

    property alias pin: pinInput.text

    function _popPinDigit() {
        pinInput.remove(pinInput.length - 1, pinInput.length)
    }

    showNavigationIndicator: false

    InfoLabel {
        //% "Enter pin code"
        text: qsTrId("qca-la-enter_pin_code")
        y: Theme.paddingLarge * 10
    }

    TextInput {
        id: pinInput
        anchors {
            bottom: keypad.top
            bottomMargin: Theme.paddingLarge
            left: keypad.left
            leftMargin: keypad._horizontalPadding
            right: backspace.left
            rightMargin: Theme.paddingMedium
        }
        horizontalAlignment: Text.AlignRight
        focus: true
        readOnly: true
        echoMode: TextInput.PasswordEchoOnEdit
        passwordCharacter: "\u2022"
        cursorDelegate: Item {}
        font.pixelSize: Theme.fontSizeHuge * 1.5
        color: Theme.highlightColor
        inputMethodHints: Qt.ImhNoPredictiveText | Qt.ImhSensitiveData | Qt.ImhHiddenText | Qt.ImhMultiLine | Qt.ImhDigitsOnly
    }

    IconButton {
        id: backspace

        anchors {
            right: keypad.right
            rightMargin: keypad._horizontalPadding
            verticalCenter: pinInput.verticalCenter
            verticalCenterOffset: Theme.paddingMedium
        }

        height: pinInput.height + Theme.paddingMedium // increase reactive area
        icon {
            source: "image://theme/icon-m-backspace-keypad"
            color: Theme.primaryColor
            highlightColor: Theme.highlightColor
        }

        opacity: pinInput.text === "" ? 0 : 1
        enabled: opacity

        Behavior on opacity { FadeAnimation {} }

        onClicked: root._popPinDigit()

        onPressAndHold: {
            root._popPinDigit()
            if (pinInput.length > 0) {
                backspaceRepeat.start()
            }
        }
        onExited: backspaceRepeat.stop()
        onReleased: backspaceRepeat.stop()
        onCanceled: backspaceRepeat.stop()
    }

    OpacityRampEffect {
        sourceItem: pinInput

        enabled: pinInput.contentWidth > pinInput.width - (offset * pinInput.width)

        direction:  OpacityRamp.RightToLeft
        slope: 1 + 6 * pinInput.width / Screen.width
        offset: 1 - 1 / slope
    }

    Keypad {
        id: keypad
        anchors {
            bottom: parent.bottom
            bottomMargin: Theme.paddingLarge * 2
        }

        pressedTextColor: Theme.colorScheme === Theme.LightOnDark ? Theme.highlightColor : Theme.highlightDimmerColor
        pressedButtonColor: Theme.rgba(Theme.highlightBackgroundColor, Theme.highlightBackgroundOpacity)
        enabled: pinInput.activeFocus
        vanityDialNumbersVisible: false
        symbolsVisible: false
        onPressed: {
            pinInput.cursorPosition = pinInput.length
            pinInput.insert(pinInput.cursorPosition, number)
        }

        BackgroundItem {
            anchors {
                left: keypad.left
                leftMargin: keypad._horizontalPadding
                bottom: keypad.bottom
            }
            width: keypad._buttonWidth
            height: keypad._buttonHeight

            Icon {
                anchors.centerIn: parent
                source: "image://theme/icon-m-back"
                color: keypad.textColor
                highlightColor: keypad.pressedTextColor
            }
            onClicked: root.reject()
        }

        BackgroundItem {
            anchors {
                right: keypad.right
                rightMargin: keypad._horizontalPadding
                bottom: keypad.bottom
            }
            width: keypad._buttonWidth
            height: keypad._buttonHeight
            visible: pinInput.length > 0

            Icon {
                anchors.centerIn: parent
                source: "image://theme/icon-m-accept"
                color: keypad.textColor
                highlightColor: keypad.pressedTextColor
            }
            onClicked: root.accept()
        }
    }

    Timer {
        id: backspaceRepeat

        interval: 150
        repeat: true

        onTriggered: {
            root._popPinDigit()
            if (pinInput.length === 0) {
                stop()
            }
        }
    }
}
