/****************************************************************************
**
** SPDX-FileCopyrightText: 2021-2025 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: Proprietary
**
****************************************************************************/
import QtQuick 2.0
import Sailfish.Silica 1.0
import ru.omp.qca 1.0
import ru.omp.qca.private 1.0
import com.jolla.settings.system 1.0
import ru.omp.authclient 1.0
import Aurora.Controls 1.0
import Sailfish.Policy 1.0

Page {
    id: root

    property alias type: firstModel.filterRole
    property alias sourceModel: firstModel.sourceModel
    readonly property bool __hasPersonal: type !== KeyStoreModel.System ? true : false
    property var requestModel

    property var secondComponentObject
    property Page _importBusyPage

    function importCertificate() {
        _importBusyPage.wait()
        KeyStoreModel.importCertificate()
    }

    function certificateName(path) {
        var fullName = (path.split("/").pop()).split(".")
        fullName.pop()
        return fullName.join(".")
    }

    function selectImportedCertificate() {
        var certificatePage = pageStack.find(function(page) { return page.hasOwnProperty('__hasPersonal') })
        var obj = pageStack.animatorPush("Sailfish.Pickers.FilePickerPage", {
                                             showSystemFiles: false,
                                             nameFilters: ['*.crt'] })
        obj.pageCompleted.connect(function(picker) {
            picker.selectedContentPropertiesChanged.connect(function() {
                var path = picker.selectedContentProperties['filePath']
                var imported = KeyStoreModel.certificateFromPEMFile(path)
                if (imported) {
                    var importDialog = pageStack.push("ru.omp.qca.private.ImportCertificateDialog", {
                                                          fileName: KeyStoreModel.importedCertificateFileName,
                                                          commonName: KeyStoreModel.importedCertificateCommonName,
                                                          acceptDestination: _importBusyPage
                                                      })
                    importDialog.accepted.connect(function() {
                        if (KeyStoreModel.importedCertificateIsRoot) {
                            query.authenticate(function() { importCertificate() }, function() { pageStack.pop(certificatePage) })
                        } else {
                            importCertificate()
                        }
                    })
                } else {
                    KeyStoreModel.resetImportedCertificate()
                    pageStack.push("ru.omp.qca.private.StatusPage", {
                                       fileName: certificateName(path),
                                       type: KeyStoreModel.Imported,
                                       errorStatus: true
                                   })
                }
            })
        })
    }

    function importKeyBundle() {
        _importBusyPage.wait()
        KeyStoreModel.importKeyBundle()
    }

    function handleKeyBundleImportResult(imported, fileName) {
        if (imported) {
            var certificatePage = pageStack.find(function(page) { return page.hasOwnProperty('__hasPersonal') })
            var importDialog = pageStack.push("ru.omp.qca.private.ImportCertificateDialog", {
                                                    fileName: KeyStoreModel.importedCertificateFileName,
                                                    commonName: KeyStoreModel.importedCertificateCommonName,
                                                    acceptDestination: _importBusyPage
                                                })
            importDialog.accepted.connect(function() {
                if (KeyStoreModel.importedCertificateIsRoot) {
                    query.authenticate(function() { importKeyBundle() }, function() { pageStack.pop(certificatePage) })
                } else {
                    importKeyBundle()
                }
            })
        } else {
            KeyStoreModel.resetImportedKeyBundle()
            pageStack.push("ru.omp.qca.private.StatusPage", {
                                fileName: fileName,
                                type: KeyStoreModel.Imported,
                                errorStatus: true,
                                error: KeyStoreModel.error
                            })
        }
    }

    function handleKeyBundlePicker(picker) {
        picker.selectedContentPropertiesChanged.connect(function() {
            var path = picker.selectedContentProperties['filePath']
            var fileName = certificateName(path)
            // Try if bundle has no password
            var imported = KeyStoreModel.keyBundleFromFile(path, "")
            if (imported) {
                handleKeyBundleImportResult(imported, fileName)
            } else if (KeyStoreModel.error == KeyStoreModel.ErrorPassphrase) {
                var passwordDialog = pageStack.push(Qt.resolvedUrl("PasswordInput.qml"))
                passwordDialog.accepted.connect(function() {
                    imported = KeyStoreModel.keyBundleFromFile(path, passwordDialog.password)
                    handleKeyBundleImportResult(imported, fileName)
                })
            } else {
                KeyStoreModel.resetImportedKeyBundle()
                pageStack.push("ru.omp.qca.private.StatusPage", {
                                fileName: fileName,
                                type: KeyStoreModel.Imported,
                                errorStatus: true,
                                error: KeyStoreModel.error
                            })
            }
        })
    }

    function selectedImportedKeyBundle() {
        var certificatePage = pageStack.find(function(page) { return page.hasOwnProperty('__hasPersonal') })
        var obj = pageStack.animatorPush("Sailfish.Pickers.FilePickerPage", {
                                             showSystemFiles: false,
                                             nameFilters: ['*.p12'] })
        obj.pageCompleted.connect(handleKeyBundlePicker)
    }

    function addCertificate() {
        switch (root.type) {
            case KeyStoreModel.SmartCard:
            pageStack.animatorPush("ru.omp.qca.private.CreateNewKeyDialog")
            break
        case KeyStoreModel.User:
            selectedImportedKeyBundle()
            break
        case KeyStoreModel.Imported:
            selectImportedCertificate()
            break
        }
    }

    function getCertificateListHeader() {
        switch (root.type) {
            case KeyStoreModel.SmartCard:
                //% "Active"
                return qsTrId("settings_system-he-active")
            case KeyStoreModel.Imported:
                //% "Others"
                return qsTrId("settings_system-he-others")
            case KeyStoreModel.User:
                //% "User"
                return qsTrId("settings_system-he-user_certificate")
            case KeyStoreModel.System:
                //% "System"
                return qsTrId("settings_system-he-system_certificate")
        }
        return ""
    }

    onStatusChanged: {
        if (status === PageStatus.Active) {
            if (_importBusyPage != null) {
                _importBusyPage.destroy()
                _importBusyPage = null
            }

            _importBusyPage = keyBusyComponent.createObject(root)
        }
    }

    CertificateModel {
        id: firstModel

        searchRoles: []
    }

    CertificateModel {
        id: secondaryModel

        filterRole:  root.type === KeyStoreModel.Imported ? KeyStoreModel.RootImported : KeyStoreModel.Request
        searchRoles: []
        sourceModel: root.type === KeyStoreModel.Imported ? KeyStoreModel : requestModel
    }

    Component {
        id: keyBusyComponent

        BusyPage {
            id: busyPage

            visibleSuccessful: false
            //% "Importing a certificate"
            labelText: qsTrId("qca-la-importing_certificate")
            onDone: {
                pageStack.push("ru.omp.qca.private.StatusPage", {
                                   fileName: KeyStoreModel.importedCertificateFileName,
                                   type: KeyStoreModel.importedCertificateIsRoot ? KeyStoreModel.RootImported : KeyStoreModel.Imported
                               })
            }
        }
    }

    Connections {
        target: KeyStoreModel
        onEntryWrittenChanged: {
            if (!isEmpty) {
                _importBusyPage.successful()
            }
        }
    }

    PageBusyIndicator {
        id: busyIndicator

        running: waitTimer.running
        z: 1

        FadeAnimator {
            target: busyIndicator
            duration: 1500
            easing.type: Easing.InExpo
            to: 1.0
        }

        // TODO needs to optimize the issuance of signals from QCA, now there is a large delay
        Timer {
            id: waitTimer
            interval: 15 * 1000
            running: true
        }
    }

    AppBar {
        id: headerBar
        visible: root.type !== KeyStoreModel.System && !busyIndicator.running

        headerText : getCertificateListHeader()

        AppBarSpacer {}

        AppBarButton {
            enabled: !placeholder.enabled
            onClicked: {
                var searchPage = pageStack.push("ru.omp.qca.CertificateSearchPage", {
                                                            "type": root.type,
                                                            "sourceModel": firstModel.sourceModel,
                                                            "requestModel": requestModel
                                                         })
            }

            icon.source: "image://theme/icon-s-search"
        }

        AppBarButton {
            id: addCertificateButton

            onClicked: addCertificate()

            visible: root.type === KeyStoreModel.User && changeUserCertificatesPolicy.value

            icon.source: "image://theme/icon-splus-add"
        }
    }

    SilicaFlickable {

        anchors {
            horizontalCenter: parent.horizontalCenter
            top: headerBar.bottom
        }

        height: parent.height - headerBar.height
        width: parent.width
        contentHeight: data.childrenRect.height

        Column {
            id: data

            anchors.horizontalCenter: parent.horizontalCenter
            width: parent.width
            spacing: Theme.paddingLarge

            Component {
                id: secondComponent

                CertificateSectionListItem {
                    onRemove: requestModel.remove(index)
                    onImporting: selectImportedCertificate()
                    onContentCountChanged: busyIndicator.running = false

                    listModel: secondaryModel
                    type: root.type === KeyStoreModel.Imported ? KeyStoreModel.RootImported : KeyStoreModel.Request
                    visible: !placeholder.enabled && !busyIndicator.running
                }
            }

            CertificateSectionListItem {
                id: certificateList

                onContentCountChanged: busyIndicator.running = false

                visible: !placeholder.enabled && !busyIndicator.running
                listModel: firstModel
                type: root.type
            }
        }
        VerticalScrollDecorator {}
    }

    Column {
        anchors.centerIn: parent
        spacing: Theme.dp(36)

        ViewPlaceholder {
            id: placeholder

            opacity: !busyIndicator.running && enabled ? 1.0 : 0.0

            //% "No certificates found"
            text: qsTrId("settings_system-no_certificates_found")

            enabled: {
                switch (root.type) {
                case KeyStoreModel.SmartCard:
                case KeyStoreModel.Imported:
                    certificateList.contentCount === 0 && secondComponentObject.contentCount === 0
                    break
                case KeyStoreModel.User:
                case KeyStoreModel.System:
                    certificateList.contentCount === 0
                    break
                default:
                    false
                    break
                }
            }
            Behavior on opacity { FadeAnimation {} }
        }

        Button {
            id: idAddCerticateButton

            anchors.horizontalCenter: parent.horizontalCenter

            onClicked: addCertificate()

            width: Theme.buttonWidthLarge
            visible: !busyIndicator.running && placeholder.enabled && changeUserCertificatesPolicy.value

            icon.source: "image://theme/icon-splus-add"
            //% "Add a certificate"
            text: qsTrId("qca-la-add_certificate")
        }
    }

    AuthQuery {
        id: query
    }

    PolicyValue {
        id: changeUserCertificatesPolicy

        policyType: PolicyValue.ChangeUserCertificatesEnabled
    }

    Component.onCompleted: {
        if (type === KeyStoreModel.SmartCard) {
            requestModel.init()
            secondComponentObject = secondComponent.createObject(data)
        }

        if (type === KeyStoreModel.Imported) {
            secondComponentObject = secondComponent.createObject(data)
        }
    }
}
