/****************************************************************************
**
** SPDX-FileCopyrightText: 2021-2025 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: Proprietary
**
****************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Pickers 1.0
import ru.omp.qca 1.0
import Aurora.Controls 1.0
import Sailfish.Policy 1.0
import "private"

Page {
    id: root

    property alias certificate: sertificateInfo.certificate
    property int type
    property string entryId
    property string exportFilePath
    property var keyStoreEntry

    signal remove()
    signal importing()

    Component {
        id: folderPickerPage

        FolderPickerPage {
            showSystemFiles: false
            //% "Export to"
            dialogTitle: qsTrId("qca-ph-export_to")
            onSelectedPathChanged: {
                waitLayer.wait()
                sertificateInfo.certificate.exportToFile(selectedPath)
                root.exportFilePath = selectedPath
            }
        }
    }

    CertificateInfoModel {
        id: sertificateInfo
    }

    Item {
        id: exportStatus
        property bool successful: false
    }

    Connections {
        target: sertificateInfo.certificate
        onFileExported: {
            exportStatus.successful = status;
            waitLayer.successful();
        }
    }

    WaitLayer {
        id: waitLayer

        visibleSuccessful: false
        onBack: waitLayer.visible = false
        onDone: pageStack.push("ru.omp.qca.private.StatusPage", {
                                   fileName: root.type === KeyStoreModel.Request ? certificate.fileName : certificate.commonName,
                                   type: root.type,
                                   isExport: true,
                                   errorStatus: !exportStatus.successful,
                                   initialPath: root.exportFilePath
                               })
    }

    AppBar {
        id: headerBar

        //% "Certificate"
        headerText: qsTrId("qca-la-certificate")

        AppBarSpacer {}

        AppBarButton {

            onClicked: popupMenu.open()
            icon.source: "image://theme/icon-splus-more"

            visible: type !== KeyStoreModel.System

            PopupMenu {
                id: popupMenu

                width: Theme.dp(528)

                PopupMenuItem {
                    onClicked: pageStack.animatorPush(Qt.resolvedUrl("CreateCertificateRequestPage.qml"), {
                                                        request: sertificateInfo.certificate,
                                                        isEdit: true
                                                    })
                    visible: type === KeyStoreModel.Request
                    //% "Edit request"
                    text: qsTrId("qca-la-edit_request")
                }

                PopupMenuItem {
                    onClicked: {
                        var removePage =  pageStack.replace("ru.omp.qca.private.RemoveDialog", {
                                                            type: type,
                                                            fileName: type === KeyStoreModel.Request ? sertificateInfo.certificate.fileName
                                                                                                    : sertificateInfo.certificate.commonName,
                                                            email: sertificateInfo.certificate.subjecEmail,
                                                            isRoot: type !== KeyStoreModel.Request ? certificate.isRoot : false,
                                                            entryId: type !== KeyStoreModel.Request ? entryId : ""
                                                        })
                        removePage.accepted.connect(function(){ remove() })
                    }
                    //% "Remove"
                    text: qsTrId("qca-la-cert_remove")
                    icon.source: "image://theme/icon-s-delete"
                    enabled: changeUserCertificatesPolicy.value
                }

                PopupMenuItem {
                    onClicked: {
                        var picker = pageStack.animatorPush(folderPickerPage)
                    }
                    visible: type !== KeyStoreModel.System
                    //% "Export"
                    text: qsTrId("qca-la-Export")
                    icon.source: "image://theme/icon-splus-export"
                }

                PopupMenuItem {
                    onClicked: importing()
                    visible: type === KeyStoreModel.Request
                    //% "Install certificate"
                    text: qsTrId("qca-la-install_certificate")
                }

                PopupMenuItem {
                    onClicked: {
                        var obj = pageStack.animatorPush("Sailfish.Pickers.FilePickerPage", {
                                                                    showSystemFiles: false,
                                                                    nameFilters: ['*.txt', '*.pdf', '*.jpg', '*.png'] })

                            obj.pageCompleted.connect(function(picker) {
                            picker.selectedContentPropertiesChanged.connect(function() {
                                var path = picker.selectedContentProperties['filePath']
                                pageStack.animatorPush("ru.omp.qca.SignVerifyPage", { path: path, keyEntry: root.keyStoreEntry })
                            })
                        })
                    }
                    visible: type === KeyStoreModel.SmartCard
                    //% "Sign file"
                    text: qsTrId("qca-la-sign_file_with_certificate")
                }
            }
        }
    }

    SilicaFlickable {
        anchors {
            horizontalCenter: parent.horizontalCenter
            top: headerBar.bottom
        }

        width: parent.width
        height: parent.height - headerBar.height

        SilicaListView {
            id: listView

            anchors.fill: parent

            footer: Item {
                height: Theme.paddingLarge
            }

            model: sertificateInfo
            section.property: "section"
            section.delegate: SectionHeader {
                text: section
            }
            
            delegate: DetailItem {
                label: model.label
                value: model.value === undefined || model.value.length === 0 ? "-" : model.value
                alignment: Qt.AlignLeft
            }
            VerticalScrollDecorator {}
        }
    }

    PolicyValue {
        id: changeUserCertificatesPolicy

        policyType: PolicyValue.ChangeUserCertificatesEnabled
    }
}
