/**
 * SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: MIT
 */

import QtQuick 2.2
import Sailfish.Silica 1.0
import QtGraphicalEffects 1.0

Item {
    id: root

    property alias iconSource: iconIndicator.source
    property int iconSize: Theme.itemSizeMedium
    property string hintText
    //% "Pull up to unlock"
    property string aliasText: qsTrId("authd-la-default_pre_auth_alias")
    property bool _showAlias: false
    property int requiredOffset

    function _initAnimation(start) {
        if (iconSource.length === 0) {
            return
        }
        startAnimation.stop()
        startAnimationText.stop()
        endAnimation.stop()
        if (start) {
            startAnimation.start()
            if (hintText.length === 0) {
                hint.opacity = 0.0
            } else if (hint.opacity == 0) {
                startAnimationText.start()
            }
        } else {
            endAnimation.start()
        }
    }

    onHintTextChanged: {
        if (hintText.length === 0) {
            startAnimationText.stop()
            hintText.opacity = 0
        } else if (hint.opacity == 0) {
            startAnimationText.start()
        }
    }

    signal startShow()
    signal endShow()

    onStartShow: _initAnimation(true)
    onEndShow: _initAnimation(false)

    Label {
        id: hint

        property int animationOffset: 0

        anchors {
            horizontalCenter: parent.horizontalCenter
            top: parent.top
            topMargin: Theme.paddingLarge - hint.animationOffset
        }

        width: parent.width - Theme.dp(144)
        opacity: 0
        color: Theme.primaryColor
        wrapMode: Text.Wrap
        text: hintText.length === 0 ? aliasText : hintText
        font.pixelSize: Theme.fontSizeSmall
        font.family: "ALS Hauss Variable"
        horizontalAlignment: Text.AlignHCenter
    }

    Item {
        id: icon

        property int oldRequiredOffset
        readonly property int requiredOffset: hint.height + Theme.paddingLarge * 2

        onRequiredOffsetChanged: {
            offsetAnimation.stop()
            offsetAnimation.from = icon.oldRequiredOffset
            offsetAnimation.to = icon.requiredOffset
            offsetAnimation.start()
            icon.oldRequiredOffset = icon.requiredOffset
        }

        anchors {
            top: parent.top
            topMargin: root.requiredOffset
            horizontalCenter: parent.horizontalCenter
        }

        width: iconSize
        height: width
        visible: iconSource.length !== 0

        NumberAnimation {
            id: offsetAnimation

            target: root
            property: "requiredOffset"
            duration: 1000
            easing.type: Easing.InOutQuad
        }

        RoundedItemAuth {
            id: mask

            anchors.centerIn: parent
            width: 0
            height: width

            Rectangle {
                anchors.fill: parent
                color: Theme.rgba("white", Theme.colorScheme === Theme.DarkOnLight ? 0.5 : 0.0)
            }

            Icon {
                id: iconIndicator

                property int animationOffset: height

                anchors {
                    top: parent.top
                    topMargin: iconIndicator.animationOffset - (icon.height - mask.height) * 0.5
                    horizontalCenter: parent.horizontalCenter
                }

                width: icon.width
                height: width
            }
        }

        Item {
            id: shadowBack

            anchors.centerIn: parent
            width: mask.width * 2
            height: width
            visible: false
            layer.enabled: true
            layer.smooth: true

            DropShadow {
                anchors.centerIn: parent
                width: mask.width
                height: width
                source: mask
                transparentBorder: true
                spread: 0.6
                samples: 33
                color: Theme.rgba(Theme.colorScheme === Theme.DarkOnLight ? "white" : "black", 0.5)
                opacity: Theme.opacityFaint
            }
        }

        Rectangle {
            anchors.centerIn: parent
            width: mask.width * 2
            height: width
            color: "transparent"
            visible: Theme.colorScheme === Theme.LightOnDark

            Rectangle {
                anchors.centerIn: parent
                width: mask.width
                height: width
                radius: width * 0.5
                color: "black"
            }

            layer.enabled: true
            layer.samplerName: "maskSource"
            layer.effect: ShaderEffect {
                property variant source: shadowBack
                fragmentShader: "
                    varying highp vec2 qt_TexCoord0;
                    uniform highp float qt_Opacity;
                    uniform lowp sampler2D source;
                    uniform lowp sampler2D maskSource;
                    void main(void) {
                        gl_FragColor = texture2D(source, qt_TexCoord0.st) * (1.0-texture2D(maskSource, qt_TexCoord0.st).a) * qt_Opacity;
                    }
                "
            }
        }
        
        MouseArea {
            anchors.fill: parent

            onClicked: {
                if (hint.opacity == 0.0 && hint.text.length !== 0) {
                    startAnimationText.start()
                }
            }
        }

        Component.onCompleted: {
            icon.oldRequiredOffset = icon.requiredOffset
            root.requiredOffset = icon.requiredOffset
        }
    }

    ParallelAnimation {
        id: startAnimationText

        NumberAnimation {
            target: hint
            property: "opacity"
            from: 0.0
            to: 1.0
            duration: 1000
            easing.type: Easing.InOutQuad
        }

        SequentialAnimation {
            loops: 2

            NumberAnimation {
                target: hint
                property: "animationOffset"
                from: 0
                to: Theme.paddingLarge
                duration: 1000
                easing.type: Easing.InOutQuad
            }

            NumberAnimation {
                target: hint
                property: "animationOffset"
                to: 0
                duration: 1000
                easing.type: Easing.InOutQuad
            }
        }
    }

    ParallelAnimation {
        id: startAnimation

        NumberAnimation {
            target: mask
            property: "width"
            from: 0.0
            to: icon.width
            duration: 1000
            easing.type: Easing.InOutQuad
        }

        NumberAnimation {
            target: iconIndicator
            property: "animationOffset"
            from: iconIndicator.height
            to: 0
            duration: 1000
            easing.type: Easing.InOutQuad
        }
    }

    ParallelAnimation {
        id: endAnimation

        NumberAnimation {
            target: hint
            property: "opacity"
            to: 0.0
            duration: 1000
            easing.type: Easing.InOutQuad
        }

        NumberAnimation {
            target: hint
            property: "animationOffset"
            to: 0
            duration: 1000
            easing.type: Easing.InOutQuad
        }

        NumberAnimation {
            target: mask
            property: "width"
            to: 0.0
            duration: 1000
            easing.type: Easing.InOutQuad
        }

        NumberAnimation {
            target: iconIndicator
            property: "animationOffset"
            to: iconIndicator.height
            duration: 1000
            easing.type: Easing.InOutQuad
        }
    }
}
