/**
 * SPDX-FileCopyrightText: 2022-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: MIT
 */

import QtQuick 2.0
import Sailfish.Silica 1.0
import com.jolla.settings.system 1.0
import ru.omp.authclient 1.0

// This component is to be used as main page in ru.auroraos.homescreen.authui app.
Item {
    id: root

    property AuthUiClient authUiClient
    readonly property bool _invisibleBottom: !authUi.emergency && loader.item
                                             && loader.item.hasOwnProperty('invisibleBottomItem')
                                             && loader.item.invisibleBottomItem
    readonly property bool _hideBottom: !authUi.emergency && loader.item && loader.item.hasOwnProperty('hideBottomItem')
                                        && loader.item.hideBottomItem
    readonly property bool _disableCancel: loader.item && loader.item.hasOwnProperty('disableCancel')
                                           && loader.item.disableCancel

    readonly property bool isTablet: Screen.sizeCategory > Screen.Medium


    readonly property int desiredButtonBoxHeight: isTablet ? Theme.dp(90) : Theme.dp(136)
    readonly property int desiredButtonHeight: Theme.itemSizeExtraSmall
    readonly property int desiredButtonTopMargin: isTablet ? Theme.paddingSmall : Theme.paddingLarge
    readonly property int desiredButtonBottomMargin: isTablet 
                                                     ? desiredButtonBoxHeight - 
                                                       desiredButtonHeight - 
                                                       desiredButtonTopMargin
                                                     : Theme.dp(40)

    // Called by compositor on display about to be on
    function show() {
        if (callUiLoader.item) {
            callUiLoader.item.canceled()
            callUiLoader.item.clear()
        }

        if (loader.item) {
            loader.item.reset()
        }
    }

    function reset() {
        // Restart auth
        authUiClient.setResponse({})
    }

    property int orientation
    focus: true

    Loader {
        id: loader

        property AuthUiClient authUi: authUiClient

        anchors {
            top: parent.top
            // It is needed for statusbar
            topMargin: Theme.paddingLarge * 2
            left: parent.left
            right: parent.right
            bottom: _hideBottom ? parent.bottom : buttonBox.top
        }

        focus: visible
        asynchronous: true
        visible: !busyIndicator.visible
        sourceComponent: authUiClient.mainPart.factor

        Connections {
            target: authUiClient

            onNextMethod: if (loader.item) loader.item.reset()
        }
    }

    BusyIndicator {
        id: busyIndicator

        anchors.centerIn: parent
        size: BusyIndicatorSize.Large
        visible: opacity > 0
        running: visible
        opacity: loader.status !== Loader.Ready ? 1.0 : 0.0
        _forceAnimation: true

        Behavior on opacity { FadeAnimation {} }
    }

    Item {
        id: buttonBox
        anchors {
            horizontalCenter: parent.horizontalCenter
            bottom: parent.bottom
        }
        height: desiredButtonBoxHeight

        Button {
            id: bottomButton

            anchors {
                horizontalCenter: parent.horizontalCenter
                bottom: parent.bottom
                bottomMargin: desiredButtonBottomMargin
                top: parent.top
                topMargin: desiredButtonTopMargin
            }

            height: desiredButtonHeight
            width: Theme.buttonWidthLarge
            text: authUiClient.textForButton
            backgroundColor: "transparent"
            enabled: authUiClient.emergency || (!authUiClient.checkInProgress && !_invisibleBottom && !_disableCancel)
            highlightBackgroundColor: Theme.rgba(Theme.primaryColor, 0.4)
            visible: !_hideBottom && !_invisibleBottom

            onClicked: {
                if (authUiClient.emergency) {
                    callUiLoader.opacity = 1.0
                } else {
                    authUiClient.cancel()
                }
            }

            Behavior on opacity { FadeAnimation {} }
            Behavior on height { FadeAnimation {} }
        }
    }

    Loader {
        id: callUiLoader

        anchors.fill: parent

        focus: opacity === 1.0
        asynchronous: true
        opacity: 0
        visible: opacity > 0
        enabled: authUiClient.emergency
        z: 2

        Behavior on opacity { FadeAnimation {} }

        sourceComponent: Component {
            EmergencyCall {
                visible: callUiLoader.visible

                onCanceled: callUiLoader.opacity = 0
            }
        }

        onOpacityChanged: if (opacity === 1.0) root.reset()
    }
}
