/**
 * SPDX-FileCopyrightText: 2022-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: MIT
 */

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0
import ru.omp.authclient 1.0

Item {
    id: root

    property AuthUiPart uiPart: authUi.mainPart
    property QtObject objPluginInfo: uiPart.info
    property int orientation: 0
    property string graphicIllustration: "graphic-locked-illustration.svg"
    property int contentHeight: 0
    property int contentWidth: 0
    readonly property real indentForTablet: (height - illustration.height - labelColumn.implicitHeight) / 7
    readonly property bool isSmartphone: LayoutClassProvider.currentDisplayClass & DisplayLayoutClass.Smartphone
    readonly property bool isLandscape: width >= height

    signal forceFocus() // this signal to compare BaseAuthPluginItem
    signal reset()
    signal clicked()
    signal tryAgain()
    signal retranslate()

    onRetranslate: {
        title.text = Qt.binding(function() {
            return objPluginInfo == null
                   //% "Unknown error"
                   ? qsTrId("auth-ui-client-la-unknown-error")
                   : objPluginInfo.resultMessage(uiPart.resultCode)
        })

        subtitle.text = Qt.binding(function() {
            return objPluginInfo != null
                   ? objPluginInfo.errorMessage(uiPart.resultCode)
                   : ""
        })
    }

    anchors.fill: parent

    function restart() {
        authUi.setResponse({})
    }

    MouseArea {
        id: mouseArea

        anchors.fill: parent
        enabled: objPluginInfo.hasOwnProperty('userLocked')
                 ? !objPluginInfo.userLocked
                 : true

        onClicked: root.restart()
    }

    Column {
        id: labelColumn

        anchors {
            top: parent.top
            topMargin: root.indentForTablet * 2
            horizontalCenter: parent.horizontalCenter
        }

        width: parent.width * (isSmartphone ? 1 : 2/3)

        Label {
            id: title

            anchors.horizontalCenter: parent.horizontalCenter
            width: parent.width - Theme.dp(36) * 2
            wrapMode: Text.Wrap
            horizontalAlignment: Text.AlignHCenter
            font.pixelSize: Theme.fontSizeLarge
            font.family: "ALS Hauss Variable"
            color: Theme.highlightColor
        }

        Label {
            id: subtitle

            anchors.horizontalCenter: parent.horizontalCenter
            width: parent.width - Theme.dp(36) * 2
            wrapMode: Text.Wrap
            horizontalAlignment: Text.AlignHCenter
            font.pixelSize: Theme.fontSizeSmall
            font.family: "ALS Hauss Variable"
            visible: text.length !== 0
        }
    }

    Icon {
        id: illustration

        anchors {
            top: labelColumn.bottom
            topMargin: root.indentForTablet * 2
            horizontalCenter: parent.horizontalCenter
        }

        color: Theme.secondaryHighlightColor
        width: Screen.sizeCategory >= Screen.Large
               ? Screen.sizeCategory === Screen.ExtraLarge
                 ? isLandscape ? Theme.dp(440) : Theme.dp(700)
                 : isLandscape ? Theme.dp(340) : Theme.dp(440)
               : Theme.dp(440)
        height: width
        source: objPluginInfo.hasOwnProperty('graphicIllustration')
                && objPluginInfo.graphicIllustration.length !== 0
                ? objPluginInfo.graphicIllustration
                : graphicIllustration
    }

    BackgroundItem {
        id: button

        anchors {
            top: illustration.bottom
            topMargin: root.indentForTablet
            horizontalCenter: parent.horizontalCenter
        }

        width: Theme.itemSizeMedium
        height: width
        highlightedColor: "transparent"
        visible: objPluginInfo.hasOwnProperty('userLocked')
                 ? !objPluginInfo.userLocked
                 : true

        onPressed: root.restart()

        Rectangle {
            anchors.fill: parent
            color: button.pressed ? Theme.highlightColor : Theme.secondaryHighlightColor
            opacity: 0.2
            radius: width * 0.5
        }

        Icon {
            anchors.centerIn: parent
            source: "image://theme/icon-splus-sync"
        }
    }

    Connections {
        target: objPluginInfo

        ignoreUnknownSignals: true

        onTryAgain: root.restart()
    }

    Component.onCompleted: retranslate()
}
