/**
 * SPDX-FileCopyrightText: 2022-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: MIT
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import com.jolla.settings.system 1.0
import org.nemomobile.systemsettings 1.0
import ru.omp.auth.sudis 1.0
import ru.omp.authclient 1.0

BaseAuthPluginItem {
    id: root

    property AuthUiPart uiPart: authUi.mainPart
    property QtObject objPluginInfo: uiPart.info

    property QtObject _feedbackEffect
    property bool _screenIsLarge: Screen.sizeCategory >= Screen.Large
    property bool showKeypad: false
    property bool useCert: true
    property bool withoutConnection: uiPart.state == SettingsSudis.EnterPasswordOrPinWithoutConnection
    property bool showUpper
    property bool showLabel: _screenIsLarge || !root.showKeypad
    property int orientation: 0
    property bool _isLandscape: orientation & Orientation.LandscapeMask
    property bool hideEmergency: showKeypad
                                 && Screen.sizeCategory == Screen.Large
                                 && _isLandscape

    function _feedback() {
        if (_feedbackEffect) {
            _feedbackEffect.play()
        }
    }

    onReset: {
        if (loaderView.item) {
            loaderView.item.reset()
        }
    }

    onRetranslate: {
        componentAuth.retranslate()
        componentLabel.retranslate()
        componentLabelOther.retranslate()
    }

    Column {
        anchors {
            top: parent.top
            topMargin: SafeZoneRect.insets.top
            horizontalCenter: parent.horizontalCenter
        }
        opacity: root.withoutConnection || (root._isLandscape && root.showKeypad) ? 0.0 : 1.0

        UserIcon {
            id: userIcon

            anchors.horizontalCenter: parent.horizontalCenter
            width: Theme.iconSizeMedium
            height: Theme.iconSizeMedium
            color: Theme.highlightColor
        }

        Label {
            anchors.horizontalCenter: parent.horizontalCenter
            text: userInfo.displayName
            color: Theme.primaryColor
            font.pixelSize: Theme.fontSizeExtraSmall
            truncationMode: TruncationMode.Fade
        }

        UserInfo {
            id: userInfo
        }
    }

    Item {
        anchors.fill: parent

        MouseArea {
            onClicked: root.showKeypad = false

            width: parent.width
            height: parent.height - (root.showKeypad && keypadLoader.item ? keypadLoader.item.height : 0)
            enabled: root.showKeypad
        }

        Column {
            anchors.verticalCenter: parent.verticalCenter
            width: parent.width

            Loader {
                id: loaderView

                width: parent.width
                sourceComponent: !root._isLandscape ? portraitViewComponent : tabletViewComponent
            }

            Item {
                width: parent.width
                height: _screenIsLarge && isPortrait ? keypadLoader.height / 2 : keypadLoader.height
                visible: root.showKeypad
            }
        }

        Loader {
            id: keypadLoader

            sourceComponent: embeddedKeyboard
            anchors.bottom: parent.bottom
            width: parent.width
            visible: root.showKeypad
        }

        Component {
            id: embeddedKeyboard

            EmbeddedKeyboard {
                onPressed: {
                    root._feedback()
                    loaderView.item.writeText(number)
                }

                width: parent.width
                showUpper: root.showUpper
            }
        }

        Button {
            id: showUpper

            onClicked: {
                root.showUpper = !root.showUpper
            }

            anchors {
                bottom: parent.bottom
                left: parent.left
            }

            width: Theme.buttonWidthTiny
            visible: root.showKeypad
            icon.source: "image://theme/icon-m-transfer"
            implicitHeight: Theme.itemSizeMedium
        }

        Button {
            onClicked: loaderView.item.enter()

            anchors {
                bottom: parent.bottom
                right: parent.right
            }

            width: Theme.buttonWidthTiny
            visible: root.showKeypad
            icon.source: "image://theme/icon-m-accept"
            implicitHeight: Theme.itemSizeMedium
        }
    }

    Component {
        id: tabletViewComponent

        Row {
            function enter() {
                componentAuth._enter()
            }

            function writeText(number) {
                componentAuth._writeText(number)
            }

            function reset() {
                componentAuth.reset()
            }

            ComponentLabel {
                id: componentLabel

                anchors.verticalCenter: parent.verticalCenter
                width: parent.width / 2
                iconSize: Theme.iconSizeExtraLarge
            }

            ComponentAuth {
                id: componentAuth

                anchors.verticalCenter: parent.verticalCenter
                width: parent.width / 2
                whiteFrameWidth: Theme.dp(768)
                _isLandscape: root._isLandscape
            }
        }
    }

    Component {
        id: portraitViewComponent

        Column {
            id: column

            function enter() {
                componentAuth._enter()
            }

            function writeText(number) {
                componentAuth._writeText(number)
            }

            function reset() {
                componentAuth.reset()
            }

            ComponentLabel {
                id: componentLabelOther

                anchors.horizontalCenter: parent.horizontalCenter
                width: parent.width
                iconSize: _screenIsLarge ? Theme.iconSizeLarge : Theme.iconSizeSmallPlus
            }

            Item {
                width: 1
                height: _screenIsLarge ? Theme.paddingLarge * 4 : Theme.paddingLarge * 2
            }

            ComponentAuth {
                id: componentAuth

                anchors.horizontalCenter: parent.horizontalCenter
                width: parent.width
                whiteFrameWidth: Screen.sizeCategory >= Screen.Large
                                    ? Theme.dp(768)
                                    : parent.width - 2 * Theme.dp(24)
            }
        }
    }
}
