/**
 * SPDX-FileCopyrightText: Copyright 2022 - 2023 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: MIT
 */

import QtQuick 2.0
import Sailfish.Silica 1.0
import QtGraphicalEffects 1.0
import Sailfish.Silica.Background 1.0

SilicaControl {
    id: dialer

    property bool isLargeScreen
    property color textColor: palette.primaryColor
    property color pressedTextColor: palette.highlightColor
    property alias pressedButtonColor: pressedButtonBackground.color
    property bool preventStealing: true
    property QtObject _feedbackEffect
    property int _maxItemsInRow: 10

    // Fill width for a small screen
    property real horizontalMargin: isLargeScreen ? Theme.horizontalPageMargin * 3 : 0

    property real _buttonWidth: ((width - horizontalMargin) - ((_maxItemsInRow - 1) * _horizontalSpacing)) / _maxItemsInRow
    property real _buttonHeight: Theme.itemSizeSmall
    property real _highlightedButtonWidth: _buttonWidth
    property real _highlightedButtonHeight: _buttonHeight
    property real _horizontalSpacing: Theme.paddingSmall * 0.5
    property real _verticalSpacing: isLargeScreen ? Theme.paddingSmall : 0
    property string _monoFontFamily: "DejaVu Sans Mono"
    property bool showUpper: false

    signal pressed(string number)
    signal released(string number)
    signal canceled(string number)
    signal clicked(string number)
    signal pressAndHold(string number, QtObject mouse)
    signal backspaceClicked()
    signal backspacePressAndHold()
    signal backspaceExited()
    signal backspaceReleased()
    signal showPasswordClicked()

    function _buttonPressed(item, number) {
        if (_feedbackEffect) {
            _feedbackEffect.play()
        }
        _centerPressedButtonBackgroundOnItem(item)
        pressedButtonBackground.visible = true
        pressedButtonShowSymbol.label = number
        dialer.pressed(number)
    }

    function _buttonReleased(number) {
        pressedButtonBackground.visible = false
        dialer.released(number)
    }

    function _buttonCanceled(number) {
        pressedButtonBackground.visible = false
        dialer.canceled(number)
    }

    function _buttonClicked(number) {
        dialer.clicked(number)
    }

    function _centerPressedButtonBackgroundOnItem(item) {
        pressedButtonBackground.x = 0
        pressedButtonBackground.y = 0
        var itemCenter = pressedButtonBackground.mapFromItem(item, item.width * 0.5, item.height * 0.5)
        pressedButtonBackground.x = itemCenter.x - pressedButtonBackground.width * 0.5
        pressedButtonBackground.y = itemCenter.y - pressedButtonBackground.height * 0.5
    }

    height: keypadColumn.implicitHeight

    KeyboardBackground {
        anchors.fill: parent
        radius: Theme.paddingSmall
    }

    Column {
        id: keypadColumn

        property Item _background: numberBackground

        anchors.horizontalCenter: parent.horizontalCenter

        width: parent.width
        spacing: _verticalSpacing

        Rectangle {
            id: numberBackground

            width: parent.width
            height: isLargeScreen ? numbersLine.height + Theme.paddingLarge : numbersLine.height + Theme.paddingSmall
            color: Theme.rgba(Theme.lightPrimaryColor, 0.1)

            EmbeddedKeyboardNumbers {
                id: numbersLine

                anchors.centerIn: parent

                buttonWidth: _buttonWidth
                buttonHeight: _buttonHeight
            }
        }

        Item {
            width: 1
            height: isLargeScreen ? Theme.paddingLarge : Theme.paddingSmall
        }

        Item {
            width: parent.width
            height: upperQWERTY.height

            EmbeddedKeyboardQWERTY {
                id: upperQWERTY

                anchors.horizontalCenter: parent.horizontalCenter
                visible: dialer.showUpper

                buttonWidth: _buttonWidth
                buttonHeight: _buttonHeight
                textColor: Theme.secondaryHighlightColor
                upperCase: true
            }

            EmbeddedKeyboardQWERTY {
                id: qwerty

                anchors.horizontalCenter: parent.horizontalCenter
                visible: !dialer.showUpper

                buttonWidth: _buttonWidth
                buttonHeight: _buttonHeight
            }
        }

        Item {
            width: 1
            height: isLargeScreen ? Theme.paddingLarge : Theme.paddingMedium
        }
    }

    Rectangle {
        id: pressedButtonBackground

        width: _highlightedButtonWidth
        height: _highlightedButtonHeight
        visible: false
        radius: Theme.paddingSmall*0.5
        // same as BackgroundItem
        color: Theme.rgba(Theme.highlightBackgroundColor, Theme.highlightBackgroundOpacity)
    }

    Rectangle {
        id: pressedButtonShowSymbol

        property alias label: pressedButtonShowLbl.text

        onVisibleChanged: {
            if (pressedButtonShowSymbol.x < 0) {
                anchors.horizontalCenterOffset = -(pressedButtonBackground.width - pressedButtonShowSymbol.width) * 0.5
            } else if ((pressedButtonShowSymbol.x + pressedButtonShowSymbol.width) > dialer.width) {
                anchors.horizontalCenterOffset = (pressedButtonBackground.width - pressedButtonShowSymbol.width) * 0.5
            } else {
                anchors.horizontalCenterOffset = 0
            }
        }

        anchors {
            verticalCenter: pressedButtonBackground.verticalCenter
            verticalCenterOffset: -height
            horizontalCenter: pressedButtonBackground.horizontalCenter
        }

        width: _highlightedButtonWidth * 1.4
        height: _highlightedButtonHeight * 0.8
        visible: pressedButtonBackground.visible
        radius: Theme.paddingSmall
        opacity: dialer._isOptionButton ? 0.0 : 1.0

        // same as BackgroundItem
        color: Theme.highlightBackgroundColor

        Label {
            id: pressedButtonShowLbl

            anchors.centerIn: parent

            font.pixelSize: Theme.fontSizeLarge
            font.bold: true
            color: dialer.textColor
        }
    }

    Component.onCompleted: {
        // Avoid hard dependency to feedback
        _feedbackEffect = Qt.createQmlObject("import QtQuick 2.0; import QtFeedback 5.0; ThemeEffect { effect: ThemeEffect.PressWeak }",
                                             dialer, 'ThemeEffect')
    }
}
