/**
 * SPDX-FileCopyrightText: 2023-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: MIT
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import com.jolla.settings.system 1.0
import org.nemomobile.systemsettings 1.0
import ru.omp.auth.sudis 1.0
import ru.omp.authclient 1.0

Column {
    property real whiteFrameWidth
    property bool _screenIsLarge: Screen.sizeCategory >= Screen.Large
    property real _sizePrimary: _screenIsLarge ? Theme.fontSizeExtraLarge
                                               : Theme.fontSizeLarge
    property real _sizeSecondary: _screenIsLarge ? Theme.fontSizeMedium
                                                 : Theme.fontSizeSmall
    property bool _isLandscape

    function _writeText(number) {
        pinInput.text = pinInput.text + number
    }

    function _popPinDigit() {
       pinInput.text = pinInput.text.substring(0, pinInput.length - 1)
    }

    function reset() {
        pinInput.text = ""
    }

    function _enter() {
        if (root.showKeypad) {
            root.showKeypad = !root.showKeypad
        }
        if (root.useCert) {
            authUi.setResponse({ "cert": root.withoutConnection || !objPluginInfo.settings.certList.length
                                      ? ""
                                      : objPluginInfo.settings.certList[cert.currentIndex],
                                "pin": pinInput.text })
        } else {
            authUi.setResponse({ "password": pinInput.text })
        }

        pinInput.text = ""
    }

    function retranslate() {
        mainLabel.text = Qt.binding(function() {
            return root.useCert
                   //% "Enter with DS"
                   ? qsTrId("authd-method-sudis-la-enter-with-ds")
                   //% "Enter with password"
                   : qsTrId("authd-method-sudis-la-enter-with-password")
        })

        smallLabel.text = Qt.binding(function() {
            return root.useCert
                   //% "Enter with password"
                   ? qsTrId("authd-method-sudis-la-enter-with-password")
                   //% "Enter with DS"
                   : qsTrId("authd-method-sudis-la-enter-with-ds")
        })

        pinInput.label = Qt.binding(function() {
            return root.useCert
                   //% "Enter pin"
                   ? qsTrId("authd-method-sudis-pa-enter-pin")
                   //% "Enter password"
                   : qsTrId("authd-method-sudis-pa-enter-password")
        })

        cert.label = Qt.binding(function() {
            return objPluginInfo.settings.certList.length > 0
                   //% "Certificate"
                   ? qsTrId("authd-method-sudis-la-certificate")
                   //% "No certificates available"
                   : qsTrId("authd-method-sudis-la-no-certificate")
        })

        enterButton.text = Qt.binding(function() {
            return busy.running
                   ? ""
                   //% "Enter"
                   : qsTrId("authd-method-sudis-la-enter")
        })
    }

    Item {
        x: (parent.width - whiteFrameWidth) / 2
        width: whiteFrameWidth
        height: frame.height

        Rectangle {
            width: parent.width
            height: parent.height
            opacity: 0.4
            color: Theme.lightPrimaryColor
        }

        Column {
            id: frame

            width: parent.width

            Item {
                width: 1
                height: Theme.paddingMedium
            }

            Label {
                id: mainLabel

                x: Theme.horizontalPageMargin
                width: parent.width - 2 * x
                color: Theme.primaryColor
                font.pixelSize: _sizePrimary
            }

            Item {
                width: 1
                height: Theme.paddingLarge * 2
            }

            Row {
                width: parent.width
                height: backspaceButton.height + Theme.paddingLarge

                PasswordField {
                    id: pinInput

                    onClicked: {
                        root.showKeypad = true
                    }

                    width: backspaceButton.visible ? parent.width - backspaceButton.width
                                                   : parent.width
                    enabled: !authUi.checkInProgress
                    showEchoModeToggle: focus

                    font.pixelSize: Theme.fontSizeLarge
                    color: Theme.highlightColor

                    softwareInputPanelEnabled: false
                    focus: root.showKeypad
                }

                Button {
                    id: backspaceButton

                    onClicked: {
                         _popPinDigit()
                    }

                    enabled: pinInput.enabled
                    visible: pinInput.length > 0
                    icon.source: "image://theme/icon-m-backspace"
                }
            }

            Item {
                width: parent.width
                height: enterButton.height + enterButton.y

                Row {
                    id: certItem

                    width: parent.width
                    height: cert.height
                    opacity: root.useCert && !root.withoutConnection ? 1.0 : 0.0

                    ComboBox {
                        id: cert

                        onDownChanged: {
                            if (root.showKeypad) {
                                root.showKeypad = !root.showKeypad
                            } else {
                                root.showLabel = _screenIsLarge
                            }
                        }

                        width: item.opacity ? parent.width - item.width - Theme.horizontalPageMargin
                                              : parent.width
                        enabled: objPluginInfo.settings.certList.length > 0 &&
                                 !authUi.checkInProgress &&
                                 !objPluginInfo.settings.certListIsUpdating

                        menu: ContextMenu {
                            Repeater {
                                model: objPluginInfo.settings.certList
                                MenuItem {
                                    text: objPluginInfo.settings.certList[index]
                                }
                            }
                        }
                    }

                    Item {
                        id: item

                        width: btnUpdateCert.width
                        height: parent.height
                        opacity: cert._menuOpen ? 0 :1

                        IconButton {
                            id: btnUpdateCert

                            onClicked: objPluginInfo.settings.updateCertList()

                            anchors.rightMargin: Theme.horizontalPageMargin

                            icon.source: "image://theme/icon-m-refresh"

                            visible: !busyFindCert.visible
                            enabled: !busy.running
                        }

                        BusyIndicator {
                            id: busyFindCert

                            visible: objPluginInfo.settings.certListIsUpdating
                            size: BusyIndicatorSize.Medium
                            running: visible
                            _forceAnimation: true
                        }
                    }
                }

                Button {
                    id: enterButton

                    onClicked: _enter()

                    x: Theme.horizontalPageMargin
                    y: (root.useCert ? certItem.height : Theme.paddingLarge) + Theme.paddingLarge
                    width: parent.width - 2 * x
                    enabled: pinInput.text != "" && !busy.running && !cert._menuOpen

                    BusyIndicator {
                        id: busy

                        anchors.centerIn: parent
                        opacity: uiPart.state == SettingsSudis.Connecting || authUi.checkInProgress ? 1.0 : 0.0
                        visible: opacity > 0
                        running: visible
                        _forceAnimation: true
                    }
                }
            }

            Item {
                width: 1
                height: Theme.paddingLarge
            }
        }
    }

    Item {
        width: 1
        height: _screenIsLarge ? Theme.paddingLarge * 2 : Theme.paddingLarge
    }

    Label {
        id: smallLabel

        property bool bottomSpaceIsSmall: _isLandscape && (root.showKeypad || cert._menuOpen)
        property bool screenIsSmall: !_screenIsLarge && root.showKeypad

        anchors.horizontalCenter: parent.horizontalCenter
        width: enterButton.width
        height: Theme.paddingLarge * 3
        wrapMode: Text.Wrap
        horizontalAlignment: Text.AlignHCenter
        enabled: !authUi.checkInProgress && !cert._menuOpen
        visible: !bottomSpaceIsSmall && !screenIsSmall
        font.pixelSize: Theme.fontSizeMedium
        color: enabled ? Theme.primaryColor : Theme.secondaryColor

        MouseArea {
            onClicked: root.useCert = !root.useCert

            anchors.fill: parent
        }
    }

    Component.onCompleted: retranslate()
}
