/**
 * SPDX-FileCopyrightText: 2023-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: MIT
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import com.jolla.settings.system 1.0
import ru.omp.authclient 1.0
import ru.omp.auth.smartcard 1.0
import QtGraphicalEffects 1.0

Item {
    id: root

    property var smartCardAttached: settingsSmartCard.attached
    property SettingsSmartCard settingsSmartCard
    property bool tryAgainState: !finishTimer.running && (uiPart.state === SettingsSmartCard.TokenNotRecognized
                                 || uiPart.state === SettingsSmartCard.ValidationFailed
                                 || uiPart.state === SettingsSmartCard.CannotBindSmartCard
                                 || uiPart.state === SettingsSmartCard.TokenNotFound
                                 || uiPart.resultCode === SettingsSmartCard.PinFailed)

    property int orientation

    readonly property bool isLandscape: orientation & Orientation.LandscapeMask

    signal findSmartCardSignal()
    signal unlockSmartCardSignal(string pin)
    signal tryAgain()
    signal finish()

    function clear() {
        pinInput.clear()
    }

    PinInput {
        id: pinInput

        anchors.fill: parent
        visible: uiPart.state === SettingsSmartCard.EnterPin ||
                 uiPart.state === SettingsSmartCard.EnterPinForced ||
                 uiPart.state === SettingsSmartCard.EnterOldPin ||
                 uiPart.state === SettingsSmartCard.EnterNewPin
        showCancelButton: false
        //% "Enter token's PIN code"
        titleText: qsTrId("auth-smartcard-la-enter-token-pin")
        subTitleText: uiPart.state === SettingsSmartCard.EnterPinForced
                      //% "You can disable the PIN code re-request in the devicelock settings"
                      ? qsTrId("auth-smartcard-la-disable-hint")
                      : settingsSmartCard.attached === SettingsSmartCard.True
                          //% "Do not extract the token"
                          ? qsTrId("auth-smartcard-la-dont-extract-token")
                          : uiPart.state === SettingsSmartCard.EnterOldPin
                              //% "Confirm the action by using current token"
                              ? qsTrId("auth-smartcard-la-confirm-action-current-token")
                              : uiPart.state === SettingsSmartCard.EnterNewPin
                                  //% "Use new token"
                                  ? qsTrId("auth-smartcard-la-use-new-token")
                                  : ""
        iconSource: "/usr/lib/qt5/qml/ru/omp/auth/smartcard/icon-smartcard.svg"
        minimumLength: settingsSmartCard.pinMinLen

        onPinConfirmed: {
            root.unlockSmartCardSignal(pinInput.enteredPin)
            pinInput.clear()
        }
    }

    Column {
        id: connectColumn

        readonly property int spacings: Math.floor((height - illustration.height - infoColumn.height) / 6)

        anchors.fill: parent
        width: parent.width
        visible: !pinInput.visible

        Item { width: parent.width; height: connectColumn.spacings }

        Column {
            id: infoColumn

            anchors.horizontalCenter: parent.horizontalCenter
            width: parent.width - Theme.paddingLarge * 2
            spacing: Theme.paddingSmall

            Label {
                anchors.horizontalCenter: parent.horizontalCenter
                width: parent.width
                horizontalAlignment: Text.AlignHCenter
                color: Theme.highlightColor
                font.pixelSize: Theme.fontSizeLarge
                wrapMode: Text.Wrap
                text: {
                    if (uiPart.resultCode == SettingsSmartCard.PinFailed) {
                        //% "Wrong pin"
                        return qsTrId("auth-smartcard-er-wrong-pin")
                    } else if (uiPart.resultCode > SettingsSmartCard.Success){
                        //% "An error occurred"
                        return qsTrId("auth-smartcard-la-error-occured")
                    }
                    switch(uiPart.state) {
                    case SettingsSmartCard.BindingSmartCard:
                        //% "The initialization process is in progress"
                        return qsTrId("auth-smartcard-la-initialization")
                    case SettingsSmartCard.BoundSmartCard:
                        //% "You can use the token to unlock the device"
                        return qsTrId("auth-smartcard-la-token-ready")
                    case SettingsSmartCard.TokenNotRecognized:
                        //% "Token not recognized"
                        return qsTrId("auth-smartcard-la-token-not-recognized")
                    case SettingsSmartCard.ValidationFailed:
                        //% "Validation failed"
                        return qsTrId("auth-smartcard-er-validation-failed")
                    case SettingsSmartCard.CannotBindSmartCard:
                        //% "Cannot bind token"
                        return qsTrId("auth-smartcard-er-cannot-bind-smartcard")
                    }
                    switch (settingsSmartCard.attached) {
                    case SettingsSmartCard.False:
                        //% "Token not found"
                        return qsTrId("auth-smartcard-la-token-not-found")
                    case SettingsSmartCard.Updating:
                        //% "Insert or tap token"
                        return qsTrId("auth-smartcard-la-insert-or-tap-token")
                    default:
                        return ""
                    }
                }
            }

            Label {
                anchors.horizontalCenter: parent.horizontalCenter
                width: parent.width
                horizontalAlignment: Text.AlignHCenter
                color: Theme.primaryColor
                font.pixelSize: Theme.fontSizeSmall
                wrapMode: Text.Wrap
                text: {
                    if (tryAgainState) {
                        //% "Try again"
                        return qsTrId("auth-smartcard-la-try-again")
                    }
                    if (uiPart.state == SettingsSmartCard.BindingSmartCard) {
                        //% "Please do not remove the token"
                        return qsTrId("auth-smartcard-la-dont-remove-token")
                    }
                    return ""
                }
            }
        }

        Item { width: parent.width; height: connectColumn.spacings * 2 }

        Icon {
            id: illustration

            anchors.horizontalCenter: parent.horizontalCenter
            width: Screen.sizeCategory === Screen.Large && isLandscape
                   ? Theme.dp(340)
                   : Screen.sizeCategory === Screen.ExtraLarge && !isLandscape 
                        ? Theme.dp(700)
                        : Theme.dp(440)
            height: width
            highlighted: true
            source: "graphic-token-illustration-"
                    + (tryAgainState || uiPart.state === SettingsSmartCard.BindingSmartCard
                       ? "error" : "ok") + ".svg"
        }

        Item {
            width: parent.width
            height: connectColumn.spacings * 3

            BusyIndicator {
                id: busyIndicator

                size: Theme.iconSizeLarge
                anchors.centerIn: parent
                running: settingsSmartCard.attached === SettingsSmartCard.Updating
                         || authUi.checkInProgress
                         || uiPart.state === SettingsSmartCard.BindingSmartCard
                visible: running
                _forceAnimation: true
            }

            Icon {
                anchors.centerIn: parent
                width: Theme.iconSizeLarge
                height: width
                visible: !busyIndicator.visible && !tryAgainButton.visible
                source: "image://theme/icon-l-checkmark"
            }

            IconButton {
                id: tryAgainButton

                anchors.centerIn: parent
                width: Theme.iconSizeLarge
                height: width
                visible: tryAgainState
                icon.source: "image://theme/icon-splus-sync"

                onClicked: {
                    if (uiPart.state === SettingsSmartCard.TokenNotFound)
                        root.findSmartCardSignal()
                    else 
                        root.tryAgain();
                }

                Rectangle {
                    anchors.fill: parent
                    radius: width / 2
                    color: Theme.rgba(Theme.highlightColor, 0.2)
                }
            }
        }
    }

    Timer {
        id: finishTimer

        onTriggered: root.finish()
        interval: 3000
    }

    MouseArea {
        anchors.fill: parent
        enabled: finishTimer.running

        onClicked: {
            finishTimer.stop()
            root.finish()
        }
    }

    Connections {
        target: uiPart

        onStateChanged: {
            if (uiPart.state === SettingsSmartCard.BoundSmartCard) {
                finishTimer.start()
            } else {
                finishTimer.stop()
            }
        }
    }
}
