/**
 * SPDX-FileCopyrightText: 2022-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: MIT
 */

import QtQuick 2.0
import ru.omp.auth.password 1.0

Item {
    id: root

    property alias settings: settingsPassword
    property alias uid: settingsPassword.uid

    signal tryAgain()

    function infoMessage(id) {
        switch (id) {
        case SettingsPassword.EnterOneTimePassword:
            //% "Enter one time password"
            return qsTrId("auth-password-la-enter-one-time-password")
        case SettingsPassword.EnterPassword:
            //% "Enter password"
            return qsTrId("auth-password-la-enter-password")
        case SettingsPassword.EnterNewPassword:
            //% "Enter a new password"
            return qsTrId("auth-password-la-enter-new-password")
        case SettingsPassword.EnterChangedPassword:
            //% "Enter changed password"
            return qsTrId("auth-password-la-enter-changed-password")
        case SettingsPassword.RetypePassword:
            //% "Retype password"
            return qsTrId("auth-password-la-retype-password")
        case SettingsPassword.SuggestNewPassword:
            //% "Want to accept the proposed password"
            return qsTrId("auth-password-la-use-password")
        case SettingsPassword.PasswordChanged:
            //% "New password set"
            return qsTrId("auth-password-la-new-password-set")
        default:
            //% "Unknown info message"
            return qsTrId("auth-password-er-unknown-info")
        }
    }

    function resultMessage(id) {
        switch (id) {
        case SettingsPassword.Success:
            //% "Success"
            return qsTrId("auth-password-er-success")
        case SettingsPassword.PasswordFailed:
            //% "Wrong password"
            return qsTrId("auth-password-er-wrong-password")
        case SettingsPassword.PasswordUpdateFailed:
            //% "Password update failed"
            return qsTrId("auth-password-er-password-update-failed")
        case SettingsPassword.PasswordsDontMatch:
            //% "Passwords don't match"
            return qsTrId("auth-password-er-passwords-dont-match")
        case SettingsPassword.UserLockedMaxAttempts:
            //% "Account is temporarily locked"
            return qsTrId("auth-password-er-account-locked-max-attempts")
        case SettingsPassword.NotSupported:
            //% "Unencrypted device is not supported"
            return qsTrId("auth-password-er-password-not-supported")
        case SettingsPassword.BadNewPassword:
            //% "New password doesn't meet policy requirements"
            return qsTrId("auth-password-er-bad-new-password")
        case SettingsPassword.PasswordMatchPrevious:
            if (settingsPassword.historyCheck.value == 1) {
                //% "New password should differ from prevous"
                return qsTrId("auth-password-er-match-previous")
            } else {
                //% "New password should differ from prevous (the number of recently checked passwords: %1)"
                return qsTrId("auth-password-er-match-previous-some").arg(settingsPassword.historyCheck.value)
            }
        case SettingsPassword.PasswordNotSet:
            //% "User cannot unlock device, ask an administrator to set one-time password"
            return qsTrId("auth-password-er-not-set")
        default:
            //% "Other error"
            return qsTrId("auth-password-er-other-error")
        }
    }

    function errorMessage(id) {
        switch (id) {
        case SettingsPassword.PasswordFailed:
        case SettingsPassword.PasswordsDontMatch:
        case SettingsPassword.PasswordUpdateFailed:
            if (settingsPassword.maxAttempts.value <= 0) {
                //% "Try again"
                return qsTrId("auth-password-er-try-again")
            }

            var attempts = settingsPassword.maxAttempts.value - settingsPassword.currentAttempts

            switch (attempts) {
            case 3:
                //% "Three attempts left"
                return qsTrId("auth-password-er-three-attempts-left")
            case 2:
                //% "Two attempts left"
                return qsTrId("auth-password-er-two-attempts-left")
            case 1:
                //% "In case of an error, the account will be temporarily blocked"
                return qsTrId("auth-password-er-wrong-password-last-chance")
            default:
                //% "Try again"
                return qsTrId("auth-password-er-try-again")
            }
        case SettingsPassword.UserLockedMaxAttempts:
            //% "Try again in %1 minutes"
            return qsTrId("auth-password-er-user-locked-try-again").arg(15)
        default:
            return ""
        }
    }

    function textForButton(id) {
        switch (id) {
        case SettingsPassword.SetLater:
            //% "Set later"
            return qsTrId("auth-password-set_later")
        }

       return ""
    }

    SettingsPassword {
        id: settingsPassword
    }
}
