/**
 * SPDX-FileCopyrightText: 2022-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: MIT
 */

import QtQuick 2.0
import QtQuick.Window 2.1 as QtQuick
import Sailfish.Silica 1.0
import ru.omp.authclient 1.0
import ru.omp.auth.password 1.0
import com.jolla.settings.system 1.0
import org.nemomobile.systemsettings 1.0

BaseAuthPluginItem {
    id: root

    property AuthUiPart uiPart: authUi.mainPart
    property QtObject objPluginInfo: uiPart.info
    property SettingsPassword settingsPassword: objPluginInfo.settings
    property int orientation: QtQuick.Window.window ? QtQuick.Window.window.contentOrientation : 0
    property bool showError: true
    property bool userIsLocked: uiPart.resultCode === SettingsPassword.UserLockedMaxAttempts
    property bool disableCancel: finishTimer.running

    signal tryAgain()

    focus: true
    contentHeight: pinInput.totalHeight + pinInput.mainContentBottomPadding
    contentWidth: pinInput.totalWidth

    onFocusChanged: if (focus) pinInput.forceActiveFocus()

    function clear() {
        if (!uiPart.params.GeneratedPassword) {
            pinInput.clear()
        }
        showError = true
    }

    function checkParams() {
        settingsPassword.externalParams = uiPart.params
        if (uiPart.params.GeneratedPassword) {
            pinInput.enteredPin = uiPart.params.GeneratedPassword
            pinInput.hidePassword = false
        }
    }

    function finish() {
        authUi.setResponse({ "finish": true })
    }

    onForceFocus: pinInput.forceActiveFocus()

    onReset: {
        if (pinInput.suggestionsEnforced) {
            return
        }
        pinInput.enteredPin = ""
        pinInput.hidePassword = true
    }

    onRetranslate: {
        pinInput.titleText = Qt.binding(function() {
            return uiPart.resultCode == 0 || !showError
                   ? objPluginInfo.infoMessage(uiPart.state)
                   : objPluginInfo.resultMessage(uiPart.resultCode)
        })

        pinInput.subTitleText = Qt.binding(function() {
            if (pinInput.isUsingOneTimePin) {
                //% "Passed to you by administrator"
                return qsTrId("auth-password-passed-by-admin")
            }

            if (pinInput.isEnterNewPin) {
                return settingsPassword.passwordRequirements
            }

            const errorMessage = objPluginInfo.errorMessage(uiPart.resultCode)
            if (showError && errorMessage.length !== 0) {
                return errorMessage
            }

            if (uiPart.params.forConfirm) {
                //% "To accept changes"
                return qsTrId("auth-password-er-to-accept-changes")
            }

            return ""
        })
    }

    Connections {
        target: uiPart

        onUpdated: root.clear()
    }

    PinInput {
        id: pinInput

        readonly property bool isEnterCurrentPin: uiPart.state === SettingsPassword.EnterPassword
        readonly property bool isEnterNewPin: uiPart.state === SettingsPassword.EnterNewPassword
        readonly property bool isUsingOneTimePin: uiPart.state === SettingsPassword.EnterOneTimePassword
        readonly property bool showPolicies: !isSimple && isEnterNewPin

        onPinConfirmed: authUi.setResponse({ 'password': enteredPin })
        onSuggestionRequested: authUi.setResponse({ 'generate': true })
        onEnteredPinChanged: {
            if (enteredPin.length !== 0) {
                showError = false
            }
            settingsPassword.policies.password = enteredPin
        }

        anchors.fill: parent
        suggestionsEnforced: uiPart.state === SettingsPassword.SuggestNewPassword
        suggestionsEnabled: suggestionsEnforced // TODO: We do not have ui for request password on keypad
        busy: !suggestionsEnforced && authUi.checkInProgress
        showCancelButton: false
        minimumLength: isEnterCurrentPin
                       ? Math.min(settingsPassword.currentLength, settingsPassword.requestedMinLength)
                       : settingsPassword.requestedMinLength
        maximumLength: settingsPassword.newLength.upperLimit
        passwordPolicies: showPolicies ? settingsPassword.policies : null
        isSimple: (isEnterCurrentPin
                   ? settingsPassword.currentStrength
                   : settingsPassword.requestedStrength) === SettingsPassword.DigitOnly
        orientation: root.orientation
        isCorrectPin: suggestionsEnforced || (showPolicies
                                              ? settingsPassword.policies.accepted
                                              : enteredPin.length >= minimumLength)
        showKeystrokes: settingsPassword.showKeystrokes
        keypadDisabled: userIsLocked || uiPart.resultCode == SettingsPassword.PasswordNotSet || finishTimer.running
        iconSource: userIsLocked
                    ? "image://theme/icon-l-high-importance"
                    : finishTimer.running ? "image://theme/icon-l-checkmark" : ""
    }

    UserInfo {
        id: userInfo
        watched: true
    }

    Connections {
        target: uiPart

        onParamsChanged: checkParams()

        onStateChanged: {
            if (uiPart.state === SettingsPassword.PasswordChanged)
                finishTimer.start()
            else
                finishTimer.stop()
        }
    }

    Timer {
        id: finishTimer

        interval: 3000

        onTriggered: root.finish()
    }

    MouseArea {
        anchors.fill: parent
        enabled: finishTimer.running

        onClicked: {
            finishTimer.stop()
            root.finish()
        }
    }

    Component.onCompleted: {
        checkParams()
        pinInput.forceActiveFocus()
        retranslate()
    }
}
