/**
 * SPDX-FileCopyrightText: 2022-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: MIT
 */

import QtQuick 2.0
import ru.omp.auth.loginrestriction 1.0

Item {
    property alias uid: loginRestriction.uid
    property alias userLocked: loginRestriction.userLocked
    property string graphicIllustration: "/usr/lib/qt5/qml/ru/omp/auth/loginrestriction/graphic-user-locked-illustration.svg"

    signal tryAgain()

    function infoMessage(id) {
        switch (id) {
        case SettingsLoginRestriction.WaitPlease:
            //% "Wait, please"
            return qsTrId("auth-loginrestriction-la-wait-please")
        default:
            //% "Unknown input request"
            return qsTrId("auth-loginrestriction-la-unknown-input-request")
        }
    }

    function resultMessage(id) {
        switch (id) {
        case SettingsLoginRestriction.Success:
            //% "Success"
            return qsTrId("auth-loginrestriction-er-success")
        default:
            //% "User account is temporarily locked"
            return qsTrId("auth-loginrestriction-er-user-temporarily-locked")
        }
    }

    function errorMessage(id) {
        switch (id) {
        case SettingsLoginRestriction.UserLocked:
            //% "The administrator has restricted access to the account"
            return qsTrId("auth-loginrestriction-er-admin-locked-user")
        case SettingsLoginRestriction.TtlBadSyntax:
            //% "The device can only be used during the set hours"
            return qsTrId("auth-loginrestriction-er-ttl-bad-syntax")
        case SettingsLoginRestriction.TtlExpired:
            //% "User account expired"
            return qsTrId("auth-loginrestriction-er-ttl-expired")
        case SettingsLoginRestriction.DayUnknown:
            //% "Cannot determine date"
            return qsTrId("auth-loginrestriction-er-day-unknown")
        case SettingsLoginRestriction.DayLocked:
            //% "The device can only be used on designated days"
            return qsTrId("auth-loginrestriction-er-day-locked")
        case SettingsLoginRestriction.TimeBadRange:
        case SettingsLoginRestriction.TimeBadSyntax:
            //% "Bad time range syntax"
            return qsTrId("auth-loginrestriction-er-time-bad-range")
        case SettingsLoginRestriction.TimeLocked:
            //% "User account locked by time of the day"
            return qsTrId("auth-loginrestriction-er-time-locked")
        case SettingsLoginRestriction.UserTempLocked:
            loginRestriction.loginLockedTill
            //% "Try again in %1"
            return qsTrId("auth-loginrestriction-er-user-locked-try-again").arg(timer.lockTimer)
        default:
            return ""
        }
    }

    Timer {
        id: timer

        property date finishTime
        property string lockTimer
        triggeredOnStart: true
        repeat: true
        onTriggered: {
            // calculate including real current date
            // bcs the Timer can have a lags and the interval will not always be exactly 1000 ms
            var diff = finishTime - Date.now()
            if (diff <= 0) {
                running = false
                lockTimer = "00:00"
                // Don't drop userLocked to false here to avoid blinking
                // "Try again" button on AuthFailed page
                tryAgain()
            } else {
                lockTimer = new Date(diff).toLocaleTimeString(Qt.locale(), "mm:ss")
            }
        }
    }

    SettingsLoginRestriction {
        id: loginRestriction

        onLoginLockedTillChanged: {
            if (loginLockedTill) {
                timer.finishTime = new Date(loginLockedTill * 1000 + 100) // add 100 ms to fix freezing
                timer.running = true
            }
        }
    }
}
