/*
 * SPDX-FileCopyrightText: Copyright 2023-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import ru.auroraos.WebView.Global 1.0
import ru.auroraos.WebView.Private 1.0

TouchInput {
    id: touchInput

    property bool contentDraggingStarted: false

    signal pressed(var touchPoints)
    signal released(var touchPoints)
    signal touchUpdated(var touchPoints)
    signal canceled(var touchPoints)

    function setOverscrollMode(newMode) {
        // Block overscroll updates during active drag-n-drop.
        // See onTouchUpdated comment for details.
        if (contentDraggingStarted === false) {
            touchArea.PullToRefresh.overscrollMode = newMode
        }
    }

    function resetOverscrollMode(mode) {
        if (touchArea.PullToRefresh.overscrollMode === mode)
            touchArea.PullToRefresh.overscrollMode = OverscrollMode.OVERSCROLL_MODE_NONE
    }

    function setContentDraggingStarted() {
        contentDraggingStarted = true
    }

    anchors.fill: parent

    MultiPointTouchArea {
        id: touchArea

        x: 0
        y: 0
        anchors.fill: parent
        maximumTouchPoints: touchInput.maximumTouchPoints
        enabled: touchInput.enabled
        mouseEnabled: true
        PullToRefresh.enabled: touchInput.parent.scrollOffsetY === 0
        PullToRefresh.refreshHandler: function() { touchInput.parent.reload() }

        Connections {
            onLoadingChanged: {
                if (!touchInput.parent.loading) {
                    touchArea.PullToRefresh.refreshCompletedCustom("image://theme/icon-s-error",
                                                                    //% "Cannot load page"
                                                                    qsTrId("browser-la-cannot_load_page"))
                }
            }

            onLoadFinished: {
                touchArea.PullToRefresh.refreshCompleted(true)
            }

            onLoadError: {
                touchArea.PullToRefresh.refreshCompletedCustom("image://theme/icon-s-error",
                                                                qsTrId("browser-la-cannot_load_page"))
            }

            target: touchInput.parent
        }

        onPressed: {
            var touchPoints = filterOutsideTouchPoints(touchPoints)

            for (var i = 0; i < touchPoints.length; i++) {
                touchInput.touchPressed(touchPoints[i].pointId, touchPoints[i].x, touchPoints[i].y)
            }
            touchInput.pressed(touchPoints)
        }

        // Release might happen outside of the MultiPointTouchArea, so we must not filter the points
        // Otherwise, CEF will continue thinking that a touch is active
        onReleased: {
            if (touchInput.contentDraggingStarted === true) {
                touchInput.contentDraggingStarted = false
                for (var i = 0; i < touchPoints.length; i++) {
                    touchInput.dragTargetDrop(touchPoints[i].x, touchPoints[i].y)
                    touchInput.dragSourceEndedAt(touchPoints[i].x, touchPoints[i].y)
                    touchInput.dragSourceSystemDragEnded();
                }
            }
            for (var i = 0; i < touchPoints.length; i++) {
                touchInput.touchReleased(touchPoints[i].pointId, touchPoints[i].x, touchPoints[i].y)
            }
            touchInput.released(touchPoints)
        }

        // Cancel might happen outside of the MultiPointTouchArea, so we must not filter the points
        // Otherwise, CEF will continue thinking that a touch is active
        onCanceled: {
            for (var i = 0; i < touchPoints.length; i++) {
                touchInput.touchCanceled(touchPoints[i].pointId, touchPoints[i].x, touchPoints[i].y)
            }
            touchInput.canceled(touchPoints)
        }

        onTouchUpdated: {
            var touchPoints = filterOutsideTouchPoints(touchPoints)

            if (touchInput.contentDraggingStarted === true) {
                for (var i = 0; i < touchPoints.length; i++) {
                    touchInput.dragTargetDragOver(touchPoints[i].x, touchPoints[i].y)
                }
                // HACK: scrolling and overscroll features are implemented
                // within webengine scope but are not in-sync between each
                // other possibly due to CEF Alloy runtime simplified design.
                // Hence touch and overscroll updates are blocked to prevent
                // undesired scrolling and overscroll during active drag-n-drop.
                // TODO: revise whether this hack is needed after M126+ upgrade
                // with migration to chrome-runtime.
                return
            }
            for (var i = 0; i < touchPoints.length; i++) {
                touchInput.touchMoved(touchPoints[i].pointId, touchPoints[i].x, touchPoints[i].y)
            }
            touchInput.touchUpdated(touchPoints)
        }

        onGestureStarted: {
            var points = filterOutsideTouchPoints(gesture.touchPoints)

            // Grab the gesture only when ALL touch points are inside
            if (!PullToRefresh.enabled && points.length === gesture.touchPoints.length) {
                gesture.grab() // We grab the gesture to prevent an overlap with system gestures like
                               // interactions with a page stack
            }
        }

        // There is a bug in Qt 5.6 that makes the MultiPointTouchArea register events outside
        // of it when it is focused.
        //
        // In order to allow proper gesture grabbing outside of WebView and event passing
        // to the engine, we need to manually check the coords of each touch point.
        //
        // returns points that are inside the MultiPointTouchArea
        function filterOutsideTouchPoints(points) {
            var filteredPoints = []

            for (var i = 0; i < points.length; i++) {
                if (points[i].x < x || points[i].x > x + width ||
                    points[i].y < y || points[i].y > y + height) {
                    continue
                }
                filteredPoints.push(points[i])
            }

            return filteredPoints
        }
    }
}
