/**
 * SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0

Row {
    id: root

    property var symbols
    property int letterSpacing
    property int symbolHeight: Theme.dp(20)
    property bool isLetters
    property bool isCaps
    readonly property int dividerWidth: Theme.dp(2)
    property int buttonWidth: Math.floor((width - dividerWidth * (symbols.count - 1)) / symbols.count)
    property int activeIndex: -1

    signal newSymbol(string symbol)

    Repeater {
        model: symbols

        Row {
            id: symbolRow

            property bool selectMode: mouseArea.pressed
            property bool animationRunning
            property int startY
            readonly property bool showTop: startY > mouseArea.mouseY + Theme.dp(15)
            readonly property string realTopSymbol: isLetters ? model.bottomSymbol.toUpperCase() : model.topSymbol
            readonly property string topSymbol: isCaps ? model.bottomSymbol : realTopSymbol
            readonly property string bottomSymbol: isCaps ? realTopSymbol : model.bottomSymbol

            height: root.height

            onShowTopChanged: {
                if (showTop) {
                    startAnimationTimer.stop()
                    animationRunning = false
                }
            }

            Item {
                anchors.verticalCenter: parent.verticalCenter
                width: buttonWidth
                height: root.height

                Column {
                    anchors.verticalCenter: parent.verticalCenter
                    width: parent.width
                    spacing: root.letterSpacing

                    Text {
                        anchors.horizontalCenter: parent.horizontalCenter
                        height: symbolHeight
                        verticalAlignment: Text.AlignVCenter
                        visible: !symbolRow.selectMode
                        lineHeight: 1.25
                        color: Theme.rgba(Theme.highlightColor, 0.8)
                        text: symbolRow.topSymbol
                        font.pixelSize: Theme.fontSizeExtraSmall
                    }

                    Text {
                        anchors.horizontalCenter: parent.horizontalCenter
                        height: symbolHeight
                        verticalAlignment: Text.AlignVCenter
                        visible: !symbolRow.selectMode
                        lineHeight: 1.25
                        color: Theme.primaryColor
                        text: symbolRow.bottomSymbol
                        font.pixelSize: Theme.fontSizeExtraLarge
                    }
                }

                Item {
                    anchors {
                        horizontalCenter: parent.horizontalCenter
                        bottom: parent.bottom
                    }

                    height: Theme.dp(264)
                    width: Theme.dp(74)
                    visible: symbolRow.selectMode

                    Rectangle {
                        anchors {
                            horizontalCenter: parent.horizontalCenter
                            top: parent.top
                            topMargin: Theme.dp(4)
                        }

                        width: parent.width - Theme.dp(8)
                        height: Theme.dp(116)
                        color: Theme._highlightBackgroundAlternativeColor
                        radius: width * 0.5

                        Rectangle {
                            id: animatedBack

                            anchors {
                                bottom: bottomCircle.bottom
                                left: bottomCircle.left
                                right: bottomCircle.right
                            }

                            radius: Math.min(width, height) * 0.5
                            visible: symbolRow.animationRunning

                            SequentialAnimation {
                                loops: Animation.Infinite
                                running: symbolRow.selectMode && symbolRow.animationRunning

                                ParallelAnimation {
                                    NumberAnimation {
                                        target: animatedBack
                                        property: "height"
                                        from: bottomCircle.height
                                        to: Theme.dp(108)
                                        duration: 500
                                        easing.type: Easing.InOutQuad
                                    }

                                    NumberAnimation {
                                        target: animatedBack
                                        property: "opacity"
                                        from: 0
                                        to: 0.5
                                        duration: 0
                                    }
                                }

                                NumberAnimation {
                                    target: animatedBack
                                    property: "opacity"
                                    from: 0.5
                                    to: 0
                                    duration: 500
                                }

                                NumberAnimation {
                                    target: animatedBack
                                    property: "opacity"
                                    from: 0
                                    to: 0
                                    duration: 500
                                }
                            }
                        }

                        ComplexPinSelectorCircle {
                            anchors {
                                top: parent.top
                                topMargin: Theme.dp(4)
                            }

                            active: symbolRow.showTop
                            symbol: symbolRow.topSymbol
                        }

                        ComplexPinSelectorCircle {
                            id: bottomCircle

                            anchors {
                                bottom: parent.bottom
                                bottomMargin: Theme.dp(4)
                            }

                            active: !symbolRow.showTop
                            symbol: symbolRow.bottomSymbol
                        }
                    }

                    Item {
                        anchors.top: parent.top
                        z: 1
                        width: parent.width
                        height: parent.width * 0.5
                        clip: true

                        Rectangle{
                            id: topHalfCircle

                            width: parent.width
                            height: parent.width
                            radius: width * 0.5
                            color: "transparent"

                            border {
                                width: dividerWidth
                                color: Theme.rgba(Theme.primaryColor, 0.4)
                            }
                        }
                    }

                    Rectangle {
                        anchors {
                            left: parent.left
                            top: parent.top
                            topMargin: parent.width * 0.5
                        }

                        z: 2
                        width: dividerWidth
                        height: Theme.dp(116)
                        gradient: Gradient {
                            GradientStop { position: 1.0; color: "transparent" }
                            GradientStop { position: 0.0; color: Theme.rgba(Theme.primaryColor, 0.4) }
                        }
                    }

                    Rectangle {
                        anchors {
                            right: parent.right
                            top: parent.top
                            topMargin: parent.width * 0.5
                        }

                        z: 2
                        width: dividerWidth
                        height: Theme.dp(116)
                        gradient: Gradient {
                            GradientStop { position: 1.0; color: "transparent" }
                            GradientStop { position: 0.0; color: Theme.rgba(Theme.primaryColor, 0.4) }
                        }
                    }
                }

                MouseArea {
                    id: mouseArea

                    anchors.fill: parent
                    hoverEnabled: visible
                    cursorShape: enabled ? Qt.PointingHandCursor : Qt.ForbiddenCursor

                    onPressedChanged: {
                        if (pressed) {
                            activeIndex = model.index
                            symbolRow.startY = mouseY
                            startAnimationTimer.start()
                        } else {
                            activeIndex = -1
                            root.newSymbol(symbolRow.showTop ? symbolRow.topSymbol : symbolRow.bottomSymbol)
                        }
                    }
                }

                Timer {
                    id: startAnimationTimer

                    interval: 500

                    onTriggered: if (!symbolRow.showTop) symbolRow.animationRunning = true
                }
            }

            Rectangle {
                id: divider

                anchors {
                    top: parent.top
                    bottom: parent.bottom
                    topMargin: Theme.dp(14)
                    bottomMargin: Theme.paddingSmall
                }

                width: visible ? dividerWidth : 0
                visible: !model.last
                gradient: Gradient {
                    GradientStop { position: 0.0; color: "transparent" }
                    GradientStop {
                        position: 1.0
                        color: buttonWidth < Theme.dp(74) && (symbolRow.selectMode || activeIndex == model.index + 1)
                               ? "transparent"
                               : Theme.rgba(Theme.highlightColor, 0.2)
                    }
                }
            }
        }
    }
}
