/*
 * SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.2
import Sailfish.Silica 1.0
import org.nemomobile.systemsettings 1.0
import ru.omp.auth.loginrestriction 1.0
import ru.omp.authclient 1.0

BackgroundItem {
    id: root

    property var user
    property bool isCurrent: false
    readonly property bool isAdmin: user.type === UserModel.DeviceOwner
    property string lastLogin: authSettings.lastLoginDate
    readonly property bool isLocked: loginRestriction.loginLocked
    readonly property bool workingHoursSet: loginRestriction.userLocked != SettingsLoginRestriction.Success

    function userInfo() {
        if (isAdmin) {
            //% "Device Administrator"
            return qsTrId("settings_users-la-admin")
        }
        if (isLocked) {
            //% "User account is locked"
            return qsTrId("settings_users-la-user_locked")
        }
        if (workingHoursSet) {
            //% "Login only during business hours"
            return qsTrId("settings_users-la-login_business_hours")
        }
        if (lastLogin === "") {
            //% "Is not auth in yet"
            return qsTrId("settings_users-la-not-auth")
        }                          
        //% "Last login is: %1"
        return qsTrId("settings_users-la-last-login").arg(lastLogin)
    }

    function retranslate() {
        mainLabel.text = Qt.binding(function() {
            return (isCurrent
                    //% "You %1 "
                    ? qsTrId("lipstick_jolla_home-la-topmenu_you").arg("\u2022")
                    : "") + user.displayName
        })
        subtitle.text = Qt.binding(function() { return root.userInfo() })
    }

    anchors.horizontalCenter: parent.horizontalCenter
    width: parent.width - Theme.paddingMedium * 2
    height: Math.max(avatar.height, infoColumn.height) + Theme.paddingMedium * 2
    highlightedColor: "transparent"

    UserAvatar {
        id: avatar

        anchors {
            left: parent.left
            verticalCenter: parent.verticalCenter
        }

        width: Theme.itemSizeSmall
        user: root.user
        highlighted: root.down
    }

    Column {
        id: infoColumn

        anchors {
            left: avatar.right
            leftMargin: Theme.paddingLarge
            right: rightArrow.left
            rightMargin: Theme.paddingMedium
            verticalCenter: parent.verticalCenter
        }

        Label {
            id: mainLabel

            width: parent.width
            wrapMode: Text.Wrap
            color: root.down ? Theme.highlightColor : Theme.primaryColor
            font.family: "ALS Hauss Variable"
            font.pixelSize: Theme.fontSizeMedium
            font.weight: Font.Medium
        }

        Row {
            spacing: Theme.paddingSmall

            Icon {
                id: lockIcon

                width: Theme.iconSizeSmall
                height: width
                visible: isLocked || workingHoursSet
                source: "image://theme/icon-s-" + (isLocked ? "lock" : "events")
                highlighted: root.down
            }

            Label {
                id: subtitle

                width: infoColumn.width - (lockIcon.visible ? lockIcon.width + Theme.paddingSmall : 0)
                color: isAdmin || root.down ? Theme.highlightColor : Theme.secondaryColor
                wrapMode: Text.Wrap
                font.pixelSize: Theme.fontSizeExtraSmall
            }
        }
    }

    Icon {
        id: rightArrow

        anchors {
            right: parent.right
            verticalCenter: parent.verticalCenter
        }

        visible: root.enabled
        source: root.enabled ? "image://theme/icon-m-right" : ""
        highlighted: root.down
    }

    SettingsLoginRestriction {
        id: loginRestriction

        uid: user.uid
    }

    AuthSettings {
        id: authSettings

        uid: user.uid
    }

    Component.onCompleted: retranslate()
}
