/*
 * SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.2
import QtGraphicalEffects 1.0
import Sailfish.Silica 1.0
import org.nemomobile.systemsettings 1.0
import ru.omp.authclient 1.0
import ru.omp.auth.loginrestriction 1.0

Item {
    id: root

    property int currentUser: 0
    property int contentWidth: parent.width
    property var _confirmUser

    readonly property int usersCount: userModel.count

    signal userConfirmed(string displayName, int uid, string login, int type)
    signal backClicked()

    function animate(src1, src2, forward) {
        animation.stop()
        animation.startItem = src1
        animation.endItem = src2
        animation.forward = forward
        animation.start()
    }

    function reset() {
        selectUserItem.opacity = 1.0
        selectUserItem.scale = 1
        confirmUserItem.opacity = 0.0
        _confirmUser = null
    }

    function retranslate() {
        selectUserItem.retranslate()
        //% "Select user"
        selectUserLabel.text = qsTrId("settings-users_select-user")
        //% "Back"
        backButton.text = qsTrId("settings-users_back")
        backFromConfirmButton.text = backButton.text
        //% "Load device for user:"
        loadDeviceLabel.text = qsTrId("settings-users_load-device-for-user")
        //% "Device Administrator"
        deviceOwnerLabel.text = qsTrId("settings_users-la-admin")
        lastLoginLabel.text = Qt.binding(function() {
            return lastLoginLabel.lastLogin === ""
                   //% "Is not auth in yet"
                   ? qsTrId("settings_users-la-not-auth")
                   //% "Last login is: %1"
                   : qsTrId("settings_users-la-last-login").arg(lastLoginLabel.lastLogin)
        })
        //% "Confirm"
        confirmButton.text = qsTrId("settings-users_confirm")
    }

    anchors.fill: parent
    enabled: opacity !== 0.0

    onOpacityChanged: if (opacity === 0.0) reset()
    onVisibleChanged: if (!visible) reset()

    Item {
        id: selectUserItem

        signal retranslate()

        anchors {
            horizontalCenter: parent.horizontalCenter
            top: parent.top
            bottom: parent.bottom
        }

        width: root.contentWidth
        opacity: 1.0
        enabled: opacity !== 0.0

        Label {
            id: selectUserLabel

            anchors {
                horizontalCenter: parent.horizontalCenter
                top: parent.top
                topMargin: Theme.paddingMedium * 4
            }

            width: parent.width
            color: Theme.highlightColor
            wrapMode: Text.Wrap
            horizontalAlignment: Text.AlignHCenter
            font.pixelSize: Theme.fontSizeLarge
        }

        SilicaListView {
            id: listView

            anchors {
                top: selectUserLabel.bottom
                topMargin: Theme.paddingMedium * 3
                bottom: backButton.top
                bottomMargin: Theme.paddingLarge
                horizontalCenter: parent.horizontalCenter
            }

            width: parent.width
            model: userModel
            clip: true
            delegate: UserSelectorUserItem {
                id: userItem

                enabled: !isLocked && !workingHoursSet && !isCurrent && AuthSettingsClient.canSwitchToUser
                user: model
                isCurrent: currentUser === model.uid

                onClicked: {
                    _confirmUser = model
                    animate(selectUserItem, confirmUserItem, true)
                }

                Connections {
                    target: selectUserItem

                    onRetranslate: userItem.retranslate()
                }
            }

            ScrollDecorator {
                flickable: listView
                color: Theme.primaryColor
            }
        }

        Button {
            id: backButton

            anchors {
                bottom: parent.bottom
                bottomMargin: Theme.paddingLarge * 1.5
                horizontalCenter: parent.horizontalCenter
            }

            width: parent.width - Theme.paddingLarge * 3
            backgroundColor: "transparent"
            border.color: confirmButton.backgroundColor

            onClicked: root.backClicked()
        }
    }

    Item {
        id: confirmUserItem

        anchors {
            horizontalCenter: parent.horizontalCenter
            bottom: parent.bottom
            top: parent.top
        }

        width: root.contentWidth
        opacity: 0.0
        enabled: opacity !== 0.0

        Label {
            id: loadDeviceLabel

            anchors {
                top: parent.top
                topMargin: Theme.paddingMedium * 4
                horizontalCenter: parent.horizontalCenter
            }

            width: parent.width
            color: Theme.highlightColor
            wrapMode: Text.Wrap
            horizontalAlignment: Text.AlignHCenter
            verticalAlignment: Text.AlignVCenter
            font.pixelSize: Theme.fontSizeLarge
        }

        Item {
            anchors {
                top: loadDeviceLabel.top
                topMargin: Theme.paddingMedium * 3
                bottom: confirmButtons.top
                horizontalCenter: parent.horizontalCenter
            }

            width: parent.width

            Column {
                anchors.verticalCenter: parent.verticalCenter
                width: parent.width

                UserAvatar {
                    id: avatar

                    anchors.horizontalCenter: parent.horizontalCenter
                    width: Theme.itemSizeHuge
                    user: _confirmUser
                }

                Item { width: parent.width; height: Theme.paddingLarge }

                Text {
                    id: userNameLabel

                    width: parent.width
                    text: _confirmUser ? _confirmUser.displayName : ""
                    color: Theme.primaryColor
                    wrapMode: Text.Wrap
                    horizontalAlignment: Text.AlignHCenter
                    verticalAlignment: Text.AlignVCenter
                    font.pixelSize: Theme.fontSizeMedium
                }

                Text {
                    id: deviceOwnerLabel

                    width: parent.width
                    visible: _confirmUser && _confirmUser.type === UserModel.DeviceOwner
                    color: Theme.highlightColor
                    horizontalAlignment: Text.AlignHCenter
                    verticalAlignment: Text.AlignVCenter
                    font.pixelSize: Theme.fontSizeExtraSmall
                }

                Item { width: parent.width; height: Theme.paddingMedium }

                Text {
                    id: lastLoginLabel

                    property string lastLogin: _confirmUser ? authSettings.lastLoginDate : ""

                    width: parent.width
                    color: Theme.secondaryColor
                    horizontalAlignment: Text.AlignHCenter
                    verticalAlignment: Text.AlignVCenter
                    font.pixelSize: Theme.fontSizeExtraSmall

                    AuthSettings {
                        id: authSettings

                        uid: _confirmUser ? _confirmUser.uid : 0
                    }
                }
            }
        }

        Column {
            id: confirmButtons

            anchors {
                bottom: parent.bottom
                bottomMargin: Theme.paddingLarge * 1.5
                left: parent.left
                right: parent.right
                leftMargin: Theme.paddingLarge * 1.5
                rightMargin: Theme.paddingLarge * 1.5
            }

            spacing: Theme.paddingLarge

            Button {
                id: backFromConfirmButton

                width: parent.width
                backgroundColor: "transparent"
                border.color: confirmButton.backgroundColor

                onClicked: animate(confirmUserItem, selectUserItem, false)
            }

            Button {
                id: confirmButton

                width: parent.width

                onClicked: root.userConfirmed(_confirmUser.displayName,
                                              _confirmUser.uid,
                                              _confirmUser.username,
                                              _confirmUser.type)
            }
        }
    }

    SequentialAnimation {
        id: animation

        property Item startItem
        property Item endItem
        property bool forward: true

        ParallelAnimation {
            NumberAnimation {
                target: animation.startItem
                property: "opacity"
                to: 0
                duration: 250
                easing.type: Easing.Linear
            }

            NumberAnimation {
                target: animation.startItem
                property: "scale"
                from: 1
                to: animation.forward ? 0.95 : 1.05
                duration: 350
                easing.type: Easing.InQuad
            }
        }

        ParallelAnimation {
            NumberAnimation {
                target: animation.endItem
                property: "opacity"
                to: 1
                duration: 250
                easing.type: Easing.Linear
            }

            NumberAnimation {
                target: animation.endItem
                property: "scale"
                from: animation.forward ? 1.05 : 0.95
                to: 1.0
                duration: 350
                easing.type: Easing.OutQuad
            }
        }
    }

    UserModel {
        id: userModel
    }

    Component.onCompleted: retranslate()
}
