/**
* SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
* SPDX-License-Identifier: Proprietary
*/

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Pickers 1.0
import Nemo.DBus 2.0
import org.nemomobile.notifications 1.0

Column {
    width: parent.width
    spacing: Theme.paddingMedium

    SectionHeader {
        //% "LGPLv3 license"
        text: qsTrId("settings_substitution-lgplv3-license")
    }

    HighlightBlock {
        Column {
            width: parent.width
            spacing: Theme.paddingMedium

            SectionItem {
                // Margins are controlled by parent
                leftMargin: 0
                rightMargin: 0
                topMargin: 0
                bottomMargin: 0

                onClicked: pageStack.push(containerPickerPage)

                //% "Substitute components"
                name: qsTrId("settings_substitution-substitute-components")
            }

            // Separator line
            Rectangle {
                anchors.horizontalCenter: parent.horizontalCenter
                width: parent.width
                height: Theme.dp(1)
                color: Theme.primaryColor
                opacity: 0.2
            }

            Row {
                width: parent.width
                spacing: Theme.paddingSmall

                Icon {
                    anchors.verticalCenter: parent.verticalCenter
                    width: Theme.iconSizeSmall
                    color: Theme.secondaryColor
                    height: Theme.iconSizeSmall
                    opacity: 0.69

                    source: "image://theme/icon-s-about"
                }

                Label {
                    anchors.verticalCenter: parent.verticalCenter
                    width: parent.width - parent.spacing - Theme.iconSizeSmall
                    color: Theme.secondaryColor
                    opacity: 0.69
                    truncationMode: TruncationMode.Fade

                    text: substitutor.isOsTainted
                        //% "Components are substituted"
                        ? qsTrId("settings_substitution-os-tainted")
                        //% "No active substitutions"
                        : qsTrId("settings_substitution-os-not-tainted")
                }
            }
        }
    }

    Component {
        id: containerPickerPage

        FilePickerPage {
            nameFilters: [ '*.scsc' ]
            onSelectedContentPropertiesChanged: {
                const page = pageStack.push(
                    Qt.resolvedUrl("SubstitutorDisclaimer.qml"),
                    {
                        "isOsTainted": substitutor.isOsTainted
                    }
                );
                page.onAccepted.connect(function () {
                    substitutor.installContainer(selectedContentProperties.filePath)
                })
            }

        }

    }

    DBusInterface {
        id: substitutor

        readonly property bool isOsTainted: getProperty('OsTainted')

        bus: DBus.SystemBus
        service: 'ru.auroraos.substitutor'
        path: '/ru/auroraos/substitutor1'
        iface: 'ru.auroraos.substitutor1'

        function installContainer(selectedFile) {
            typedCall(
                'InstallContainer',
                { 'type': 's', 'value': selectedFile },
                function(result) { console.log('call completed with:', result) },
                function(error, message) {
                    console.log('call failed', error, 'message:', message)

                    //% "System"
                    notification.appName = qsTrId("settings_substitution-system")

                    if (error == "org.freedesktop.DBus.Error.InvalidFileContent") {
                        //% "Container processing error"
                        notification.body = qsTrId("settings_substitution-error-processing")
                        //% "Container contains invalid components for replacement"
                        notification.summary = qsTrId("settings_substitution-error-invalid")
                    } else if (error == "ru.auroraos.substitutor1.Error.ContainerRejected") {
                        notification.body = qsTrId("settings_substitution-error-processing")
                        //% "Incorrect format"
                        notification.summary = qsTrId("settings_substitution-error-incorrect")
                    } else {
                        //% "Container installation internal error"
                        notification.body = qsTrId("settings_substitution-error-internal")
                    }
                    notification.publish()
                    pageStack.pop()
                }
            )
        }
    }

    Notification {
        id: notification

        isTransient: true
        appIcon: "icon-lock-warning"
        urgency: Notification.Critical
    }
}
