/**
 * SPDX-FileCopyrightText: 2013-2020 Jolla Ltd
 * SPDX-FileCopyrightText: 2021-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.0
import org.nemomobile.configuration 1.0
import Sailfish.Silica 1.0
import Sailfish.Telephony 1.0
import MeeGo.QOfono 0.2
import ru.omp.authclient 1.0

Item {
    id: root

    property alias simManager: pinInput.simManager
    property alias requestedPinType: pinInput.requestedPinType
    property alias multiSimManager: pinInput.multiSimManager
    property alias retrying: pinInput.retrying
    property alias enteredPin: pinInput.enteredPin
    property alias busy: pinInput.busy
    property alias _currentPinType: pinInput._currentPinType
    property alias titleText: pinInput.titleText
    property bool showCancel
    property bool enteringNewPin
    property string oldPin
    property string newPin
    property string badPinWarning
    property bool showEmergency: true
    //% "Re-entered PIN did not match."
    readonly property string pinMismatchText: qsTrId("settings_pin-he-reentered_pin_mismatch")

    signal pinConfirmed()
    signal pinEntryCanceled()

    function requestAndConfirmNewPin() {
        if (oldPin.length == 0) {
            enteringNewPin = true
            oldPin = enteredPin
        } else if (newPin.length == 0) {
            newPin = enteredPin
        } else if (newPin == enteredPin) {
            enteringNewPin = false
            root.pinConfirmed()
        } else {
            badPinWarning = pinMismatchText
        }

        pinInput.clear()
    }

    function clear() {
        pinInput.clear()
    }

    onRetryingChanged: {
        oldPin = ""
        newPin = ""
    }

    anchors.fill: parent

    Item {
        anchors {
            top: parent.top
            bottom: bottomButton.top
            bottomMargin: Theme.paddingLarge
        }

        width: parent.width

        PinInput {
            id: pinInput

            property OfonoSimManager simManager
            property int requestedPinType
            property bool retrying
            property string modemPath: simManager.modemPath
            property bool permanentlyLocked: false
            property var multiSimManager
            property string _currentPinType: root.enteringNewPin && simManager.isPukType(requestedPinType)
                                             ? simManager.pukToPin(requestedPinType)
                                             : requestedPinType
            property int simIndex: multiSimManager && multiSimManager.valid
                                   ? multiSimManager.indexOfModem(simManager.modemPath) + 1
                                   : 1
            property bool pinNeedSkipped

            anchors.fill: parent

            keypadDisabled: permanentlyLocked
            minimumLength: simManager.minimumPinLength(_currentPinType)
            maximumLength: simManager.maximumPinLength(_currentPinType)
            iconSource: permanentlyLocked
                        ? "image://theme/icon-l-high-importance?" + Theme.highlightColor
                        : pinNeedSkipped ? "image://theme/icon-l-high-importance"
                                         : "image://theme/icon-l-sim-" + simIndex + "?" + Theme.highlightColor
            titleText: {
                if (pinNeedSkipped) {
                    //% "SIM card not used"
                    return qsTrId("settings_pin-la-sim_not_used")
                }
                if (permanentlyLocked) {
                    //% "SIM card %1 is blocked"
                    return qsTrId("settings_pin-la-permanently_locked").arg(simIndex)
                }

                if (root.enteringNewPin) {
                    if (root.newPin == "") {
                        //% "Enter new PIN"
                        return qsTrId("settings_pin-he-enter_new_pin")
                    }

                    //% "Re-enter new PIN"
                    return qsTrId("settings_pin-he-reenter_new_pin")
                }

                switch (requestedPinType) {
                case OfonoSimManager.SimPin:
                    return retrying
                            //: Displayed when the PIN code has been entered incorrectly
                            //% "Incorrect PIN code"
                            ? qsTrId("settings_pin-he-wrong_pin")
                            //: Request the user to enter a PIN code for SIM access
                            //% "Enter PIN code"
                            : qsTrId("settings_pin-he-enter_pin")
                case OfonoSimManager.SimPuk:
                    return retrying
                            //: Displayed when the PUK code has been entered incorrectly
                            //% "Incorrect PUK code"
                            ? qsTrId("settings_pin-he-wrong_puk")
                            //: Request the user to enter a PUK code for SIM access
                            //% "Enter PUK code"
                            : qsTrId("settings_pin-he-enter_puk")
                default:
                    console.log("SimPinInput: unrecognized PIN/PUK type:", requestedPinType)
                    return ""
                }
            }

            subTitleText: {
                if (pinNeedSkipped) {
                    //% "Only emergency calls are possible. You can unlock the SIM card in Settings | PIN code"
                    return qsTrId("settings_pin-la-sim_warning")
                }

                if (permanentlyLocked) {
                    //% "Contact your service provider"
                    return qsTrId("settings_pin-la-permanently_locked_hint")
                }

                if (root.badPinWarning.length > 0) {
                    return root.badPinWarning
                }

                var simName = multiSimManager && multiSimManager.valid
                    ? multiSimManager.simNames[simIndex - 1].replace("|", "•")
                    : ""

                if (root.enteringNewPin) {
                    return simName
                }

                switch (requestedPinType) {
                case OfonoSimManager.SimPin:
                    var pinRetries = simManager.pinRetries[requestedPinType]
                    if (isNaN(pinRetries) || pinRetries === 0) {
                        return simName
                    }

                    return (pinRetries === 1)
                            //: Warns that the PUK code will be required if this last PIN attempt is incorrect.
                            //% "Only 1 attempt left. If this goes wrong, SIM will be blocked with PUK code."
                            ? qsTrId("settings_pin-la-last_pin_warning")
                            //: Warns about the number of retries available for PIN input
                            //% "%n attempts left"
                            : qsTrId("settings_pin-la-pin_warning", pinRetries)
                case OfonoSimManager.SimPuk:
                    var pukRetries = simManager.pinRetries[requestedPinType]
                    if (pukRetries === 0) {
                        permanentlyLocked = true
                        return simName
                    }
                    return (pukRetries === 1)
                            //: Warns that this is the last available PUK code attempt.
                            //% "Only 1 attempt left. If this goes wrong, SIM card will be permanently locked."
                            ? qsTrId("settings_pin-la-last_puk_warning")    
                            : (pukRetries === undefined)
                                //: Warns that the device has been locked with a PUK code.
                                //% "SIM locked with PUK code. Contact your network service provider for the PUK code."
                            ? qsTrId("settings_pin-la-puk_warning_attempts")
                                //: Warns that the device has been locked with a PUK code (%1 = number of attempts remaining before SIM is permanently locked)
                                //% "%n attempts left. Contact your network service provider for the PUK code."
                            : qsTrId("settings_pin-la-puk_warning_attempts_with_retries", pukRetries)
                default:
                    console.log("SimPinInput: unrecognized PIN/PUK type:", requestedPinType)
                    return simName
                }
            }

            onCanceled: {
                if (pinInput.pinNeedSkipped) {
                    root.pinEntryCanceled()
                    return
                }

                pinInput.pinNeedSkipped = true
            }

            onPinConfirmed: {
                pinInput.pinNeedSkipped = false
                if (enteringNewPin) {
                    requestAndConfirmNewPin()
                } else {
                    root.pinConfirmed()
                }
            }
        }
    }

    Button {
        id: bottomButton

        anchors {
            horizontalCenter: parent.horizontalCenter
            bottom: parent.bottom
            bottomMargin: Theme.dp(40)
        }

        visible: showEmergency

        width: Theme.buttonWidthLarge
        text: {
            if (showCancel)
                //% "Skip"
                return qsTrId("skip-call-button")

            return pinInput.pinNeedSkipped
                   //% "Skip"
                   ? qsTrId("skip-call-button")
                   //% "Emergency call"
                   : qsTrId("emergency-call-button")
        }
        backgroundColor: "transparent"
        highlightBackgroundColor: Theme.rgba(Theme.primaryColor, 0.4)

        onClicked: {
            if (showCancel || pinInput.pinNeedSkipped) {
                root.pinEntryCanceled()
                return
            }
            if (pinInput.permanentlyLocked) {
                pinInput.permanentlyLocked = false
                pinInput.onPinConfirmed()
                return
            }
            callUiLoader.opacity = 1.0
            if (callUiLoader.item) {
                callUiLoader.item.forceFocus()
            }
        }

        Behavior on opacity { FadeAnimation {} }
        Behavior on height { FadeAnimation {} }
    }

    Loader {
        id: callUiLoader

        anchors.fill: parent
        focus: opacity === 1.0
        asynchronous: true
        opacity: 0
        visible: opacity > 0
        sourceComponent: EmergencyCall {
            onCanceled: callUiLoader.opacity = 0
        }

        onItemChanged: if (item) item.canceled.connect(function(){ callUiLoader.opacity = 0 })

        Behavior on opacity { FadeAnimation {} }
    }
}
