/*
 * SPDX-FileCopyrightText: 2021-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.0
import Sailfish.Silica 1.0

Item {
    id: root

    property alias name: titleLabel.text
    property alias nameHeader: headerLabel.text
    property alias help: helpLabel.text
    property bool showHeader
    property bool showDescriptionLabel: true
    property bool highlightDescription
    property alias description: descriptionLabel.text
    property url pageSource
    property bool needWrapToTitle
    property var args

    property real leftMargin: Theme.horizontalPageMargin
    property real rightMargin: Theme.horizontalPageMargin
    property real topMargin: Theme.paddingMedium * 2
    property real bottomMargin: Theme.paddingMedium * 2

    signal clicked()

    width: parent.width
    height: contentHeader.implicitHeight

    Column {
        id: contentHeader

        width: parent.width

        SectionHeader {
            id: headerLabel

            visible: showHeader
            text: nameHeader
        }

        Label {
            id: helpLabel

            x: Theme.paddingLarge
            width: parent.width - x * 2
            visible: showHeader
            text: help
            font.pixelSize: Theme.fontSizeExtraSmall
            wrapMode: Text.Wrap
            color: Theme.secondaryColor
        }

        Item {
            width: 1
            height: Theme.paddingMedium
            visible: showHeader
        }

        Item {
            width: parent.width
            // icon is placed right to content, so it might be (and actually is) the limiting factor in height
            height: Math.max(content.implicitHeight, icon.height)
            opacity: root.enabled ? 1.0 : 0.4

            Column {
                id: content

                anchors {
                    left: parent.left
                    leftMargin: root.leftMargin
                    right: icon.left
                    rightMargin: Theme.paddingMedium
                    verticalCenter: parent.verticalCenter
                }

                Item {
                    width: parent.width
                    height: root.topMargin
                }

                Label {
                    id: titleLabel

                    width: parent.width

                    truncationMode: TruncationMode.Fade
                    color: mouseArea.pressed ? Theme.highlightColor : Theme.primaryColor
                    wrapMode: needWrapToTitle ? Text.Wrap : Text.NoWrap
                    font.pixelSize: Theme.fontSizeMedium
                }

                Label {
                    id: descriptionLabel

                    width: parent.width
                    wrapMode: Text.Wrap
                    color: mouseArea.pressed || highlightDescription
                           ? Theme.secondaryHighlightColor
                           : Theme.secondaryColor
                    visible: text.length !== 0 && showDescriptionLabel
                    font.pixelSize: Theme.fontSizeExtraSmall
                }

                Item {
                    width: parent.width
                    height: root.bottomMargin
                }
            }

            Image {
                id: icon

                anchors {
                    right: parent.right
                    rightMargin: root.rightMargin
                    verticalCenter: content.verticalCenter
                }

                source: "image://theme/icon-m-right?"
                        + (mouseArea.pressed ? Theme.highlightColor : Theme.primaryColor)
            }

            MouseArea {
                id: mouseArea

                anchors.fill: parent
                cursorShape: Qt.PointingHandCursor

                onClicked: {
                    if (pageSource.toString() !== "") {
                        var obj = pageStack.animatorPush(pageSource.toString(), args)
                        if (!obj.pageCompleted) {
                            return
                        }
                        obj.pageCompleted.connect(function(page) {
                            if (page.onArgsChanged) {
                                page.onArgsChanged.connect(function() { args = page.args })
                            }
                        })
                    }
                    root.clicked()
                }
            }
        }
    }
}
