/**
 * SPDX-FileCopyrightText: 2013-2020 Jolla Ltd
 * SPDX-FileCopyrightText: 2019-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import QtFeedback 5.0
import "private"

Item {
    id: root

    property string subTitleText
    property string titleText
    property string enteredPin
    property int minimumLength: 0
    property int maximumLength: 20
    property bool hidePassword: true
    property bool showCancelButton: true
    property bool suggestionsEnforced
    property bool suggestionsEnabled
    property var passwordPolicies: null
    property string iconSource
    property bool busy
    property bool keypadDisabled
    property bool isSimple: true
    property bool showKeystrokes: true
    property int orientation: __silica_applicationwindow_instance.orientation
    property bool isCorrectPin: minimumLength <= enteredPin.length
    property int totalWidth: Math.min(parent.width, 
                                      isSimple
                                      ? Theme.dp(528)
                                      : parent.width - (isTablet
                                                          ? isLandscape && parent.width > Theme.dp(1100)
                                                            ? Theme.dp(120)
                                                            : Theme.paddingLarge
                                                          : Theme.paddingSmall * 2) * 2)

    readonly property int mainContentBottomPadding: isSimple && isTablet && !isLandscape
                                                    ? (height - backgroungRect.height) / 5
                                                    : 0
    readonly property bool showPassword: enteredPin.length !== 0 && !suggestionsEnforced
    readonly property bool isLandscape: parent.width > parent.height
    readonly property bool isScreenLandscape: orientation & Orientation.LandscapeMask
    // for one time password in user creation dialog
    readonly property bool isTablet: parent.width > Theme.dp(1100)
                                     || (Screen._isTablet && isScreenLandscape === isLandscape)
    readonly property int sizeCategory: isTablet !== Screen._isTablet ? Screen.Medium : Screen.sizeCategory
    readonly property int totalHeight: Math.min(parent.height,
                                                isSimple
                                                ? sizeCategory === Screen.Large && isLandscape
                                                  ? Theme.dp(756)
                                                  : Theme.dp(830)
                                                : sizeCategory === Screen.ExtraLarge
                                                  ? isLandscape ? Theme.dp(900) : Theme.dp(1100)
                                                  : sizeCategory === Screen.Large
                                                    ? isLandscape ? Theme.dp(756) : Theme.dp(900)
                                                    : Theme.dp(830))
    readonly property string specialChars: "!@#$%?&*-+"

    signal pinConfirmed()
    signal canceled()
    signal suggestionRequested()

    function checkSuggestions() {
        if (suggestionsEnforced && !isCorrectPin) {
            root.suggestionRequested()
        }
    }

    function clear() {
        enteredPin = ""
        checkSuggestions()
    }

    function handlePressed(value) {
        if (value === "back") {
            feedbackEffect.play()
            if (showPassword) {
                showLastDigitTimer.stop()
                enteredPin = enteredPin.substring(0, enteredPin.length - 1)
                complexPin.dropCaps()
            } else {
                root.canceled()
            }
            return
        }
        if (value === "ok") {
            feedbackEffect.play()
            root.pinConfirmed()
            return
        }
        if (maximumLength <= 0 || enteredPin.length < maximumLength) {
            feedbackEffect.play()
            enteredPin += value
            complexPin.dropCaps()
        }
    }

    function isLetter(c) {
        return c.toLowerCase() != c.toUpperCase() && c.charCodeAt(0) <= 127
    }

    function isDigit(c) {
        return c >= '0' && c <= '9'
    }

    anchors.fill: parent
    focus: true

    Keys.onPressed: {
        if (event.key === Qt.Key_Backspace) {
            if (showCancelButton || enteredPin.length !== 0) {
                handlePressed("back")
            }
            return
        }
        if (event.key === Qt.Key_Enter || event.key === Qt.Key_Return) {
            if (isCorrectPin) {
                handlePressed("ok")
            }
            return
        }        

        if ((isSimple && isDigit(event.text))
            || (!isSimple && (isDigit(event.text) || isLetter(event.text) || specialChars.indexOf(c) >= 0))) {
            handlePressed(event.text)
        }
    }

    onSuggestionsEnforcedChanged: checkSuggestions()
    onIsCorrectPinChanged: checkSuggestions()

    ComplexPinBackgroundRectangle {
        anchors {
            bottom: backgroungRect.top
            bottomMargin: isTablet
                          ? isLandscape
                            ? sizeCategory === Screen.Large ? Theme.paddingMedium : Theme.paddingLarge
                            : Theme.paddingLarge
                          : Theme.paddingMedium
        }

        width: totalWidth
        height: Math.min(parent.height - backgroungRect.height - Theme.paddingMedium * 2,
                         policiesGrid.height + hints.vertialMargins * 2)
        visible: !isSimple && passwordPolicies && passwordPolicies.count > 0
        isTablet: root.isTablet

        SilicaFlickable {
            id: hints

            readonly property bool isLandscapeTablet: isLandscape && isTablet
            readonly property bool isSmallLandscape: sizeCategory === Screen.Large && isLandscape
            readonly property int horizontalMargins: isTablet ? Theme.dp(48) : Theme.dp(30)
            readonly property int vertialMargins: isSmallLandscape ? Theme.dp(30) : horizontalMargins
            readonly property int policyWidth: passwordPolicies && isLandscapeTablet
                                               ? (width - Theme.paddingMedium) * 0.5 - horizontalMargins
                                               : width
            readonly property int firstColSize: passwordPolicies
                                                ? isLandscapeTablet
                                                  ? passwordPolicies.count * 0.5 + passwordPolicies.count % 2
                                                  : passwordPolicies.count
                                                : 0

            anchors {
                fill: parent
                topMargin: vertialMargins
                bottomMargin: vertialMargins
                leftMargin: horizontalMargins
                rightMargin: horizontalMargins
            }

            contentHeight: policiesGrid.height
            clip: true

            ScrollDecorator {
                flickable: hints
                color: Theme.primaryColor
            }

            Row {
                id: policiesGrid

                readonly property int multiColWidth: width / 2 - Theme.paddingMedium

                width: parent.width
                height: Math.max(hintsSecondColumn.height, hintsMainColumn.height)
                spacing: Theme.paddingMedium

                ComplexPinHintsColumn {
                    id: hintsMainColumn

                    function isCorrectIndex(index) {
                        return index < hints.firstColSize
                    }

                    width: hints.isLandscapeTablet ? parent.width * 0.5 - Theme.paddingMedium : parent.width
                    passwordPolicies: root.passwordPolicies
                    isSmallSize: !isTablet || hints.isSmallLandscape
                    policyWidth: hints.policyWidth
                }

                ComplexPinHintsColumn {
                    id: hintsSecondColumn

                    function isCorrectIndex(index) {
                        return index >= hints.firstColSize
                    }

                    visible: hints.isLandscapeTablet
                    width: hintsMainColumn.width
                    passwordPolicies: root.passwordPolicies
                    isSmallSize: !isTablet || hints.isSmallLandscape
                    policyWidth: hints.policyWidth
                }
            }
        }
    }

    ComplexPinBackgroundRectangle {
        id: backgroungRect

        anchors {
            bottom: parent.bottom
            bottomMargin: mainContentBottomPadding
        }

        width: totalWidth
        height: {
            var keypad = isSimple ? simplePin : complexPin
            const keypadHeight = keypad.height + keypad.anchors.bottomMargin
            const minInfoHeight = Math.max(passwordColumn.height, titlesColumn.height) + Theme.paddingLarge * 2
            const hintsHeight = policiesGrid.height + (hints.vertialMargins + Theme.paddingMedium) * 2
                                + Theme.paddingLarge * (isTablet ? 2 : 1)
            const requestedHeight = parent.height - hintsHeight
            const minimumHeight = keypadHeight + minInfoHeight
            if (requestedHeight > totalHeight) {
                return totalHeight
            }
            return Math.max(minimumHeight, requestedHeight)
        }
        isTablet: root.isTablet
    }

    Item {
        anchors {
            top: backgroungRect.top
            bottom: isSimple ? simplePin.top : complexPin.top
            left: parent.left
            right: parent.right
            leftMargin: Theme.dp(36)
            rightMargin: Theme.dp(36)
        }

        MouseArea {
            anchors.fill: infoColumn
            enabled: passwordColumn.visible
            hoverEnabled: enabled
            cursorShape: Qt.PointingHandCursor

            onClicked: {
                hidePassword = !hidePassword
                root.forceActiveFocus()
            }

            onContainsMouseChanged: console.log("test contains:", containsMouse)
        }

        Column {
            id: infoColumn

            anchors {
                left: parent.left
                right:parent.right
                centerIn: parent
            }

            BusyIndicator {
                anchors.horizontalCenter: parent.horizontalCenter
                width: Theme.iconSizeLarge
                running: root.busy
                visible: running
                _forceAnimation: true
            }

            Column {
                id: passwordColumn

                visible: showPassword && !root.busy

                Label {
                    id: passwordLabel

                    anchors.horizontalCenter: parent.horizontalCenter
                    text: enteredPin
                    visible: !hidePassword
                    color: Theme.highlightColor
                    font.pixelSize: Theme.fontSizeExtraLarge
                }

                Row {
                    id: hiddenPassword

                    property bool showLastDigit
                    property int curLength: enteredPin.length
                    property int prevLength: 0

                    height: Math.max(lastDigitText.height, Theme.dp(14))
                    visible: hidePassword
                    spacing: Theme.paddingSmall

                    onCurLengthChanged: {
                        if (curLength === prevLength + 1) {
                            prevLength = curLength
                            hiddenPassword.showLastDigit = true
                            showLastDigitTimer.start()
                            return
                        }
                        prevLength = curLength
                        hiddenPassword.showLastDigit = false
                    }

                    Repeater {
                        anchors.horizontalCenter: parent.horizontalCenter
                        model: enteredPin.length + (hiddenPassword.showLastDigit ? -1 : 0)
                        delegate: Rectangle {
                            anchors.verticalCenter: parent.verticalCenter
                            width: Theme.dp(14)
                            height: width
                            radius: width * 0.5
                            color: Theme.highlightColor
                        }
                    }

                    Text {
                        id: lastDigitText

                        text: enteredPin.length === 0 ? "" : enteredPin[enteredPin.length - 1]
                        visible: hiddenPassword.showLastDigit
                        verticalAlignment: Text.AlignVCenter
                        color: Theme.highlightColor
                        font.pixelSize: Theme.fontSizeExtraLarge
                    }

                    Timer {
                        id: showLastDigitTimer

                        interval: 1000

                        onTriggered: hiddenPassword.showLastDigit = false
                    }
                }
            }

            Column {
                id: titlesColumn

                spacing: Theme.paddingSmall
                visible: !showPassword && !root.busy

                Item {
                    anchors {
                        margins: Theme.paddingMedium
                        horizontalCenter: parent.horizontalCenter
                    }

                    width: Theme.iconSizeLarge
                    height: width
                    visible: root.iconSource !== "" || suggestionsEnforced

                    Icon {
                        anchors.fill: parent
                        source: suggestionsEnforced ? "icons/m-pin-enforced.svg" : root.iconSource
                        color: Theme.highlightColor
                    }
                }

                Label {
                    id: titleLabel

                    anchors.horizontalCenter: parent.horizontalCenter
                    width: backgroungRect.width - 2 * Theme.paddingMedium
                    color: Theme.highlightColor
                    wrapMode: Text.Wrap
                    horizontalAlignment: Text.AlignHCenter
                    visible: text.length !== 0
                    text: suggestionsEnforced
                          ? //% "You must use the suggested password"
                            qsTrId("settings-pin-la-you-must-use-suggested-password")
                          : titleText
                    font.pixelSize: Theme.fontSizeLarge
                }

                Label {
                    anchors.horizontalCenter: parent.horizontalCenter
                    width: titleLabel.width
                    color: Theme.primaryColor
                    wrapMode: Text.Wrap
                    horizontalAlignment: Text.AlignHCenter
                    visible: text.length !== 0
                    text: suggestionsEnforced
                          ? //% "We recommend that you do not write it down or store it in an accessible place"
                            qsTrId("settings-pin-la-suggestion-recommendation")
                          : subTitleText
                    font.pixelSize: Theme.fontSizeSmall
                }
            }
        }
    }

    Column {
        id: complexPin

        property bool isCaps
        property bool isStrongCaps
        readonly property int margins: isTablet ? Theme.paddingLarge : Theme.dp(18)
        readonly property int letterSpacing: sizeCategory === Screen.ExtraLarge ? Theme.dp(28) : Theme.dp(22)
        readonly property int rowHeight: sizeCategory === Screen.Large
                                         ? Theme.dp(94)
                                         : sizeCategory === Screen.ExtraLarge ? Theme.dp(112) : Theme.dp(106)

        function dropCaps() {
            if (!isStrongCaps) {
                isCaps = false
            }
        }

        function capsClicked() {
            if (isStrongCaps) {
                isStrongCaps = false
                isCaps = false
            } else if (isCaps) {
                isStrongCaps = true
            } else {
                isCaps = true
            }
        }

        anchors {
            bottom: backgroungRect.bottom
            bottomMargin: complexPin.margins
            horizontalCenter: parent.horizontalCenter
        }

        width: totalWidth
        enabled: !root.busy && !keypadDisabled && visible
        opacity: enabled ? 1.0 : Theme.opacityHigh
        visible: !isSimple && !suggestionsEnforced
        spacing: Theme.dp(2)

        ComplexPinRow {
            height: complexPin.rowHeight
            width: parent.width
            letterSpacing: complexPin.letterSpacing
            isCaps: complexPin.isCaps
            symbols: ListModel {
                ListElement { topSymbol: "!"; bottomSymbol: "1" }
                ListElement { topSymbol: "@"; bottomSymbol: "2" }
                ListElement { topSymbol: "#"; bottomSymbol: "3" }
                ListElement { topSymbol: "$"; bottomSymbol: "4" }
                ListElement { topSymbol: "%"; bottomSymbol: "5" }
                ListElement { topSymbol: "?"; bottomSymbol: "6" }
                ListElement { topSymbol: "&"; bottomSymbol: "7" }
                ListElement { topSymbol: "*"; bottomSymbol: "8" }
                ListElement { topSymbol: "-"; bottomSymbol: "9" }
                ListElement { topSymbol: "+"; bottomSymbol: "0"; last: true }
            }

            onNewSymbol: handlePressed(symbol)
        }

        ComplexPinRow {
            id: secondComplexRow

            height: complexPin.rowHeight
            width: parent.width
            letterSpacing: complexPin.letterSpacing
            isLetters: true
            isCaps: complexPin.isCaps
            symbols: ListModel {
                ListElement { bottomSymbol: "q" }
                ListElement { bottomSymbol: "w" }
                ListElement { bottomSymbol: "e" }
                ListElement { bottomSymbol: "r" }
                ListElement { bottomSymbol: "t" }
                ListElement { bottomSymbol: "y" }
                ListElement { bottomSymbol: "u" }
                ListElement { bottomSymbol: "i" }
                ListElement { bottomSymbol: "o" }
                ListElement { bottomSymbol: "p"; last: true }
            }

            onNewSymbol: handlePressed(symbol)
        }

        Row {
            anchors.left: parent.left
            height: complexPin.rowHeight
            spacing: isTablet ? Theme.dp(2) : 0

            ComplexPinRow {
                height: parent.height
                buttonWidth: secondComplexRow.buttonWidth
                letterSpacing: complexPin.letterSpacing
                isLetters: true
                isCaps: complexPin.isCaps
                symbols: ListModel {
                    ListElement { bottomSymbol: "a" }
                    ListElement { bottomSymbol: "s" }
                    ListElement { bottomSymbol: "d" }
                    ListElement { bottomSymbol: "f" }
                    ListElement { bottomSymbol: "g" }
                    ListElement { bottomSymbol: "h" }
                    ListElement { bottomSymbol: "j" }
                    ListElement { bottomSymbol: "k" }
                    ListElement { bottomSymbol: "l" }
                }

                onNewSymbol: handlePressed(symbol)                
            }

            Item {
                height: parent.height
                width: (secondComplexRow.buttonWidth - complexPinRemoveButton.width) / 2 - Theme.dp(2)
                visible: isTablet
            }

            ComplexPinActionButton {
                id: complexPinRemoveButton

                width: isTablet 
                        ? isLandscape 
                          ? height 
                          : secondComplexRow.buttonWidth - Theme.paddingSmall * 2 - Theme.dp(2)
                        : secondComplexRow.buttonWidth + Theme.paddingSmall * 2
                contentWidth: isTablet 
                               ? isLandscape 
                                 ? height 
                                 : secondComplexRow.buttonWidth - Theme.paddingSmall * 2 - Theme.dp(2) * 2
                               : secondComplexRow.buttonWidth - Theme.dp(2)
                enabled: showPassword
                isLandscape: root.isLandscape
                isTablet: root.isTablet
                sizeCategory: root.sizeCategory
                iconSource: "icons/m-backspace-alt.svg"

                onClicked: handlePressed("back")
            }
        }

        Row {
            id: bottomRow

            readonly property int maxActionButtonWidth: (totalWidth - bottomButtonsRow.width) * 0.5 - Theme.dp(4)
            readonly property int actionButtonWidth: (totalWidth - bottomButtonsRow.width) * 0.5 
                                                    - (Theme.paddingSmall + Theme.paddingLarge)

            anchors.horizontalCenter: parent.horizontalCenter
            height: complexPin.rowHeight
            spacing: Theme.dp(2)

            ComplexPinActionButton {
                id: leftButton
                
                defaultActive: complexPin.isCaps || complexPin.isStrongCaps
                isLandscape: root.isLandscape
                isTablet: root.isTablet
                sizeCategory: root.sizeCategory
                iconSource: "icons/m-pin-shift-"
                            + (complexPin.isStrongCaps ? "on-strong" : complexPin.isCaps ? "on" : "off") + ".svg"

                onClicked: complexPin.capsClicked()

                Component.onCompleted: {
                    if (isTablet)
                        leftButton.width = Qt.binding(function() { return bottomRow.actionButtonWidth; })
                    else
                        leftButton.maxWidth = Qt.binding(function() { return bottomRow.maxActionButtonWidth; })
                }
            }

            ComplexPinRow {
                id: bottomButtonsRow

                height: parent.height
                buttonWidth: secondComplexRow.buttonWidth
                letterSpacing: complexPin.letterSpacing
                isLetters: true
                isCaps: complexPin.isCaps
                symbols: ListModel {
                    ListElement { bottomSymbol: "z" }
                    ListElement { bottomSymbol: "x" }
                    ListElement { bottomSymbol: "c" }
                    ListElement { bottomSymbol: "v" }
                    ListElement { bottomSymbol: "b" }
                    ListElement { bottomSymbol: "n" }
                    ListElement { bottomSymbol: "m"; last: true }
                }

                onNewSymbol: handlePressed(symbol)
            }

            ComplexPinActionButton {
                id: rightButton

                defaultActive: true
                isLandscape: root.isLandscape
                isTablet: root.isTablet
                sizeCategory: root.sizeCategory
                iconSource: "icons/m-accept-bold.svg"
                enabled: isCorrectPin

                Component.onCompleted: {
                   if (isTablet)
                        rightButton.width =  Qt.binding(function() { return bottomRow.actionButtonWidth; })
                    else
                        rightButton.maxWidth = Qt.binding(function() { return bottomRow.maxActionButtonWidth; } )
                }

                onClicked: handlePressed("ok")
            }
        }
    }

    Item {
        id: simplePin

        anchors {
            bottom: backgroungRect.bottom
            bottomMargin: Theme.dp(56)
            horizontalCenter: parent.horizontalCenter
        }

        height: simplePinGrid.height
        width: totalWidth
        enabled: !root.busy && !keypadDisabled && visible
        opacity: enabled ? 1.0 : Theme.opacityHigh
        visible: isSimple || suggestionsEnforced

        Item {
            id: passwordSuggestor

            anchors.fill: parent
            visible: suggestionsEnforced
            enabled: visible

            Item {
                anchors {
                    top: parent.top
                    bottom: acceptButton.top
                    left: parent.left
                    right: parent.right
                }

                Column {
                    anchors.centerIn: parent

                    Label {
                        anchors.horizontalCenter: parent.horizontalCenter
                        text: enteredPin
                        color: Theme.highlightColor
                        font.pixelSize: Theme.fontSizeExtraLarge
                    }

                    IconButton {
                        anchors.horizontalCenter: parent.horizontalCenter
                        icon.source: "image://theme/icon-m-sync"

                        onClicked: root.suggestionRequested()
                    }
                }
            }

            BackgroundItem {
                id: acceptButton

                anchors {
                    bottom: parent.bottom
                    leftMargin: Theme.paddingLarge
                    rightMargin: Theme.paddingLarge
                    horizontalCenter: parent.horizontalCenter
                }

                highlightedColor: "transparent"
                enabled: suggestionsEnforced && isCorrectPin
                opacity: enabled ? 1.0 : Theme.opacityHigh
                hoverEnabled: visible
                cursorShape: enabled ? Qt.PointingHandCursor : Qt.ForbiddenCursor

                onClicked: handlePressed("ok")

                Text {
                    anchors.centerIn: parent
                    //% "Use this password"
                    text: qsTrId("settings-pin-la-use-this-password")
                    color: acceptButton.down ? Theme.highlightColor : Theme.primaryColor
                    font.pixelSize: Theme.fontSizeMedium
                }
            }
        }

        Grid {
            id: simplePinGrid

            property bool needRemovedAll

            anchors {
                bottom: parent.bottom
                horizontalCenter: parent.horizontalCenter
            }

            columns: 3
            columnSpacing: Theme.dp(40)
            rowSpacing: Theme.paddingSmall
            visible: !suggestionsEnforced
            enabled: visible

            Repeater {
                model: ListModel {
                    ListElement { value: "1" }
                    ListElement { value: "2" }
                    ListElement { value: "3" }
                    ListElement { value: "4" }
                    ListElement { value: "5" }
                    ListElement { value: "6" }
                    ListElement { value: "7" }
                    ListElement { value: "8" }
                    ListElement { value: "9" }
                    ListElement { value: "back" }
                    ListElement { value: "0" }
                    ListElement { value: "ok" }
                }

                BackgroundItem {
                    id: key

                    property bool showDown

                    height: Theme.itemSizeLarge
                    width: Theme.itemSizeLarge
                    highlightedColor: "transparent"
                    enabled: model.value.length === 1 ||
                             (model.value == "ok" && isCorrectPin) ||
                             (model.value == "back" && (showCancelButton || enteredPin.length !== 0))
                    opacity: enabled ? 1.0 : Theme.opacityHigh
                    hoverEnabled: visible
                    cursorShape: enabled ? Qt.PointingHandCursor : Qt.ForbiddenCursor

                    onPressedChanged: simplePinGrid.needRemovedAll = pressed && model.value == "back"

                    onClicked: {
                        showDown = true
                        downTimer.start()
                        handlePressed(model.value)
                    }

                    Label {
                        anchors.centerIn: parent
                        visible: model.value !== "back" && model.value !== "ok"
                        text: model.value
                        color: Theme.primaryColor
                        font.pixelSize: Theme.dp(46)
                        font.family: "ALS Hauss Variable"
                    }

                    Icon {
                        anchors.centerIn: parent
                        width: Theme.itemSizeLarge / 1.75
                        height: width
                        visible: model.value === "back" || model.value === "ok"
                        source: "icons/m-" + (model.value === "ok"
                                              ? "accept-bold"
                                              : showPassword ? "backspace-alt" : "cancel-bold") + ".svg"
                    }

                    Rectangle {
                        anchors.fill: parent
                        color: showKeystrokes && (showDown || key.down) ? Theme.highlightColor : "transparent"
                        opacity: 0.4
                        radius: width * 0.5
                    }

                    Timer {
                        id: downTimer

                        interval: 50

                        onTriggered: showDown = false
                    }
                }
            }
        }
    }

    Timer {
        interval: 100
        repeat : true
        running: (simplePinGrid.needRemovedAll || complexPinRemoveButton.pressed) && enteredPin.length > 0

        onTriggered: handlePressed("back")
    }

    ThemeEffect {
        id: feedbackEffect

        effect: ThemeEffect.PressWeak
    }

    Component.onCompleted: {
        checkSuggestions()
        forceActiveFocus()
    }
}
